/** SPI Flash Filesystem Example
 * Opens file in filesystem, prints contents of the file, and writes to the file.
 *
 */
#include "mbed.h"
#include "SpiFlash25.h"
#include "spiffs.h"
#include <string>

// this value represents the number of files you can have open at the same time
// adjust it according to your requirements
#define MAX_CONCURRENT_FDS      2

#define PAGE_SIZE               256
#define SECTOR_SIZE             64*1024
#define MEM_SIZE                2*1024*1024

static u8_t spiffs_work_buf[PAGE_SIZE * 2];
static u8_t spiffs_fds[32 * MAX_CONCURRENT_FDS];
static u8_t spiffs_cache_buf[(PAGE_SIZE + 32) * 4];

static SpiFlash25 flash(PB_5, PB_4, PB_3, PB_0);
static spiffs fs;
static spiffs_config cfg;

// glue code between SPI driver and filesystem driver
int spi_read(unsigned int addr, unsigned int size, unsigned char* data) {
    if (flash.read(addr, size, (char*)data))
        return SPIFFS_OK;
    return -1;
}
int spi_write(unsigned int addr, unsigned int size, unsigned char* data) {
    if (flash.write(addr, size, (const char*)data))
        return SPIFFS_OK;
    return -1;
}
int spi_erase(unsigned int addr, unsigned int size) {
    flash.clear_sector(addr);
    return SPIFFS_OK;
}

int main(void) {
    int ret;
    int handle;
    char data[256];
    char msg[] = "Hello World!\r\n";
    char file[] = "test.txt";
    string sdata;

    // configure the filesystem
    cfg.phys_size = MEM_SIZE;
    cfg.phys_addr = 0;
    cfg.phys_erase_block = SECTOR_SIZE;
    cfg.log_block_size = SECTOR_SIZE;
    cfg.log_page_size = PAGE_SIZE;

    cfg.hal_read_f = &spi_read;
    cfg.hal_write_f = &spi_write;
    cfg.hal_erase_f = &spi_erase;

    printf("\r\n\r\nSPI flash example started.\r\n\r\n");

    // erase entire flash
    // THIS WILL ERASE THE ENTIRE FLASH! EVERYTHING ON IT WILL BE LOST AND CANNOT BE RECOVERED!
    flash.clear_mem();

    printf("Mounting SPI flash file system ... ");
    // mount the filesystem
    ret = SPIFFS_mount(&fs, &cfg, spiffs_work_buf, spiffs_fds, sizeof(spiffs_fds), spiffs_cache_buf, sizeof(spiffs_cache_buf), NULL);
    if (ret) {
        printf("Failed.\r\n");
        return ret;
    }
    else
        printf("OK.\r\n");

    // write to the file
    printf("Opening file 'test.txt' for writing ... ");
    handle = SPIFFS_open(&fs, file, SPIFFS_CREAT | SPIFFS_RDWR | SPIFFS_APPEND, 0);
    if (handle < 0) {
        printf("Failed.\r\n");
        return SPIFFS_errno(&fs);
    }
    else
        printf("OK.\r\n");

    printf("Writing data to the file ... ");
    if (handle) {
        ret = SPIFFS_write(&fs, handle, msg, sizeof(msg));
        if (ret < 0) {
            printf("Failed.\r\n");
            return SPIFFS_errno(&fs);
        }
        else
            printf("OK. \r\nWrote %d bytes to the file.\r\n", ret);
        SPIFFS_close(&fs, handle);
        printf("File closed.\r\n");
    }

    // read the current file contents
    printf("Getting size of 'test.tx' file ... ");
    spiffs_stat stat;
    memset(&stat, 0, sizeof(stat));
    ret = SPIFFS_stat(&fs, file, &stat);
    if (ret) {
        printf("Failed.\r\n");
        return SPIFFS_errno(&fs);
    }
    else
        printf("OK.\r\nFile size: %d bytes.\r\n", stat.size);

    printf("Opening file 'text.txt' for reading ... ");
    handle = SPIFFS_open(&fs, file, SPIFFS_RDWR, 0);
    if (handle < 0) {
        printf("Failed.\r\n");
        return SPIFFS_errno(&fs);
    }
    else
        printf("OK.\r\n");

    printf("Reading data from the file ... ");
    if (handle) {
        while (sdata.size() < stat.size) {
            ret = SPIFFS_read(&fs, handle, data, stat.size - sdata.size() < sizeof(data) ? stat.size - sdata.size() : sizeof(data));
            if (ret < 0) {
                printf("\r\nRead failed. Error: %d\r\n", SPIFFS_errno(&fs));
                continue;
            }
            printf("OK.\r\nRead %d bytes.\r\n", ret);
            sdata.append(data, ret);
        }

        printf("File content: ");
        for (int i = 0; i < sdata.size(); i++)
            printf("%c", sdata[i]);
        printf("\r\n");

        SPIFFS_close(&fs, handle);
        printf("File closed.\r\n");
    }

    printf("\r\nSPI flash example finished\r\n\r\n");

    return 0;
}