#!/bin/bash

# Mbed tool for the QtCreator IDE.

# --------------------------------------------------------------------
# Default settings. You can modify them according to your system.
MBED_SYS_PATH=~/src/arm/sys
LIB_PATH=~/src/arm/lib
OS=6
PROFILE=DEBUG
PROJ_NAME=${PWD##*/}
PORT=/dev/ttyACM0

# --------------------------------------------------------------------
# Mbed command parser. Sets bash variables based on the selected Mbed CLI options.
# Ones parsing is complete the options and arguments can be passed to Mbed CLI or other programs.
MACROS=()
SRC_LIB_PATH=()
while [[ $# -gt 0 ]]
do
    key="$1" # Get the first token of the command line

    case $key in
        all|-f) # Flash target with binary.
        FLASH='-f'
        shift # past argument
        ;;

        -c|clean) # Clean files before building.
        CLEAN='-c'
        shift # past argument
        ;;


        -D|--macro) # Add macro.
        MACROS+=('-D '$2)
        shift # past argument
        shift # past value
        ;;

        -l) # Add library with relative path.
        SRC_LIB_PATH+=' --source '${LIB_PATH}/$2
        shift # past argument
        shift # past value
        ;;

        -m) # Define target Micro-Controller Unit (MCU).
        MCU="$2"
        shift # past argument
        shift # past value
        ;;

        -os) # Select Mbed Operating System (2 or other).
        OS="$2"
        shift # past argument
        shift # past value
        ;;

        -p|--port) # Select serial port.
        PORT="$2"
        shift # past argument
        shift # past value
        ;;

        -r|--release) # Select release build profile
        PROFILE=RELEASE
        shift # past argument
        ;;

        *) # Take additional options and pass them to Mbed CLI.
        ARGS+=("$key ")
        shift # past argument
        ;;
    esac
done

# --------------------------------------------------------------------
# Set path to the selected Mbed OS tools and profile.
if [ ${OS} == 2 ]
then
	GCC_ARM_PATH=/opt/gcc-arm-none-eabi-6-2017-q2-update
	MBED_OS_PATH=${MBED_SYS_PATH}/mbed
	CUSTOM_TARGETS_PATH=${MBED_SYS_PATH}/custom_targets/mbed
	PROF=debug
	if [ ${PROFILE} == RELEASE ]
	then
	    PROF=release
	fi
else
	GCC_ARM_PATH=/opt/gcc-arm-none-eabi-9-2019-q4-major
	MBED_OS_PATH=${MBED_SYS_PATH}/mbed-os
	CUSTOM_TARGETS_PATH=${MBED_SYS_PATH}/custom_targets/mbed-os
	PROF=${MBED_SYS_PATH}/mbed-os/tools/profiles/debug.json
	if [ ${PROFILE} == RELEASE ]
	then
	    PROF=${MBED_SYS_PATH}/mbed-os/tools/profiles/release.json
	fi
fi

# --------------------------------------------------------------------
# Call Mbed CLI "config" command to set path to the toolchain (GCC ARM) and Mbed OS (2 or other)
mbed config -G GCC_ARM_PATH ${GCC_ARM_PATH}/bin
mbed config -G MBED_OS_DIR ${MBED_OS_PATH}

# --------------------------------------------------------------------
# Create a new project if no such exists yet.
if ! [ -f ./.mbed ]
then
    mbed new .
fi

# --------------------------------------------------------------------
# Execute Mbed CLI "export" command using the selected options.
mbed export -i qtcreator -m ${MCU} --source . --source ${CUSTOM_TARGETS_PATH} --source ${MBED_OS_PATH} --source ${GCC_ARM_PATH}/arm-none-eabi/include ${SRC_LIB_PATH}

# --------------------------------------------------------------------
# Add additional files to the project to be available in the QtCreator IDE for editing.
find ~/.local/bin -name "mbed.bat" >> ${PROJ_NAME}.files
find . -name "*.json" >> ${PROJ_NAME}.files
find . -name "*.mbedignore"  >> ${PROJ_NAME}.files

# --------------------------------------------------------------------
# Execute Mbed CLI "compile" command using the selected options.
mbed compile -t GCC_ARM -m ${MCU} ${ARGS} --source . --source ${CUSTOM_TARGETS_PATH} --source ${MBED_OS_PATH} ${SRC_LIB_PATH} $CLEAN --profile ${PROF} |tee ./BUILD/compile_output.txt

# Check for a binary built by Mbed CLI.
RESULT=$(grep "Elf2Bin" ./BUILD/compile_output.txt)

# --------------------------------------------------------------------
# The following commands are for flashing some selected targets.
# You can delete (or ignore) the whole section below if you prefer to flash your targets using the "drag & drop" or other methods.

# If the compilation succeeded (a binary exists) flash the target MCU with the resulted binary file.
if [ "${RESULT}" != "" ]
then
    # Uncomment the following lines if you want to build a "hex" file for Flash Magic.
	#if [ "${MCU}" == "LPC1768DBG" ]
	#then
	#	${GCC_ARM_PATH}/bin/arm-none-eabi-objcopy -I binary -O ihex ./BUILD/${MCU}/GCC_ARM-${PROFILE}/${PROJ_NAME}.bin ./BUILD/${MCU}/GCC_ARM-${PROFILE}/${PROJ_NAME}.hex
	#fi

    # If the "all" or "-f" option was selected choose the right tool and configuration.
    if [ "${FLASH}" == "-f" ]
	then
		case ${MCU} in
		    # Targets supported by pyOCD
			LPC1768|LPC1768DBG)
			    MCU_CFG="pyOCD"
				;;
				# Targets supported by OpenOCD.
			# Configure OpenOCD according to the selected target.
			BLUEPILL|MAPLE_MINI|NUCLEO_F103RB|STM32F103RC|STM32F103VE)
				MCU_CFG="/usr/local/share/openocd/scripts/board/st_nucleo_f103rb.cfg"
				;;
			NUCLEO_F411RE|NUCLEO_F446RE|STM32F407VE)
				MCU_CFG="/usr/local/share/openocd/scripts/board/st_nucleo_f4.cfg"
				;;
			NUCLEO_F767ZI)
				MCU_CFG="/usr/local/share/openocd/scripts/board/stm32f7discovery.cfg"
				;;
				# Warn when flashing of selected target MCU is not supported.
			*)
				MCU_CFG="not_supported" 
				echo "No support to flash "${MCU}
				;;
		esac

        # When the target MCU is supported flash the target MCU with the binary code.
		if [ "$MCU_CFG" != "not_supported" ]
		then
			echo "Flashing the "${MCU}" ..."
			case ${MCU_CFG} in
			    # Use pyOCD to flash the target when port was specified.
				pyOCD)
					if [ "$PORT" == "/dev/ttyACM0" ]
					then
					    # Use pyOCD to flash the target when port was specified.
						~/.local/bin/pyocd-flashtool -t lpc1768 ./BUILD/${MCU}/GCC_ARM-${PROFILE}/${PROJ_NAME}.bin
					else
					    # Use "lpc2isp" otherwise.
						lpc21isp -bin ./BUILD/${MCU}/GCC_ARM-${PROFILE}/${PROJ_NAME}.bin ${PORT} 115200 96000
					fi
					;;
				# Use OpenOCD to flash the target MCU
				*)
				    openocd -f ${MCU_CFG} -c """program ./BUILD/${MCU}/GCC_ARM-${PROFILE}/${PROJ_NAME}.elf verify reset exit"""
	  				;;
	  		esac
			echo "Done."
	  	fi
	fi
fi
