/*
 * An example of using the Menu library - Simple Menu system with push buttons.
 *
 * The Menu library was created to facilitate designs with a display and push buttons.
 * - number of menus and push buttons is practically not limited by the software
 * - function of individual push buttons varies depending on the selected menu
 *
 *    ---------------------------------
 *   |                                  |
 *   |   ----------------------------   |
 *   |  |                            |  |
 *   |  |                            |  |
 *   |  |                            |  |
 *   |  |         Display            |  |
 *   |  |                            |  |
 *   |  |                            |  |
 *   |  |                            |  |
 *   |   ----------------------------   |
 *   |       O      O     O     O       |
 *   |      Btn1   Btn2  Btn3  Btn4     |
 *   |                                  |
 *    ----------------------------------
 *
 * Each push button must be connected to an mbed digital input pin and the ground.
 *
 *
 *     -------------------------
 *    |
 *    |      mbed board
 *    |
 *     -------------------------
 *        | Input pin
 *        |
 *        o
 *         / Button        ...
 *        o
 *        |
 *       ---
 *       GND
 *
 *
 * NOTE: When creating a MenuSystem the constructor automatically connects
 * an internal pull-up resistor to each PushButton input used by the MenuSytem.
 */

#include "mbed.h"
#include "Menu.h"

Serial pc(USBTX, USBRX);

/*
 * Variables used in this example
 */
int  setpoint;
int* p_setpoint;
int  loSetpoint = 20;
int  hiSetpoint = 50;

/*
 * Define push buttons to be used in menu.
 */
#if defined(TARGET_LPC1768)
PushButton btnMode(p21), btnMore(p22), btnLess(p23), btnSave(p24);
#else
PushButton btnMode(D2), btnMore(D3), btnLess(D4), btnSave(D5);
#endif


/*
 * Define an array of PushButton pointers initialized with push buttons created above.
 */
PushButton* buttons[] = {&btnMode, &btnMore, &btnLess, &btnSave};


/*
 * Create a menu system object associated with the push buttons
 */
MenuSystem  menuSystem(buttons, 4 /*, 300 */); // name of the array, number of buttons, (debounce time in ms)

/*
 * Create as many menus as you need.
 *
 * In this example we create the following three menus:
 * 1. Main
 * 2. Set high setpoint
 * 3. Set low setpoint
 *
 * NOTE: The function of individual push buttons will vary depending on the actually opened menu.
 */
Menu  menuMain(menuSystem);
Menu  menuSetHiSetpoint(menuSystem);
Menu  menuSetLoSetpoint(menuSystem);

/*
 * Declare all functions to be called on 'button pressed' events
 */
void openMenu_Main(void);
void openMenu_SetHiSetpoint(void);
void openMenu_SetLoSetpoint(void);
void incrementSetpoint(void);
void decrementSetpoint(void);
void saveSetpoint(void);

/**
 * @brief
 * @note
 * @param
 * @retval
 */
int main(void) {

    // Attach 'button pressed' event handler functions to the individual push buttons.
    // NOTE: It is not compulsory to attach a function to all push buttons in each menu.
    //

    // 'Main' menu:
    menuMain.attach(buttons[0], openMenu_SetHiSetpoint);

    //
    // 'Set High Setpoint' menu:
    menuSetHiSetpoint.attach(buttons[0], openMenu_SetLoSetpoint);
    menuSetHiSetpoint.attach(buttons[1], incrementSetpoint);
    menuSetHiSetpoint.attach(buttons[2], decrementSetpoint);
    menuSetHiSetpoint.attach(buttons[3], saveSetpoint);

    //
    // 'Set Low Setpoint' menu:
    menuSetLoSetpoint.attach(buttons[0], openMenu_Main);
    menuSetLoSetpoint.attach(buttons[1], incrementSetpoint);
    menuSetLoSetpoint.attach(buttons[2], decrementSetpoint);
    menuSetLoSetpoint.attach(buttons[3], saveSetpoint);

    openMenu_Main();    // open 'Main' menu

    while(1) {
        // When a button is pressed a flag is set immediately via an interrupt service routine.
        // But the assigned handler is called later on here by the 'handleButtons()' function.
        menuSystem.handleButtons();

        //.. Your main code
    }
}

/*
 * Lets define te menu functions:

 * NOTE: For the sake of simplicity, in this example we do not connect any display to the mbed.
 *       Instead we'll utilize a serial terminal running on the connected PC.
 */

/**
 * @brief
 * @note    None
 * @param   None
 * @retval  None
 */
void openMenu_Main(void) {
    menuSystem.open(menuMain);  // activate the menu
    pc.printf("---------------------------------------------------\r\n");
    pc.printf("Main menu:\r\n");
    pc.printf("Press 'Mode' to open 'High setpoint' menu.\r\n");
    pc.printf("hiSetpoint = %i\r\n", hiSetpoint);
    pc.printf("loSetpoint = %i\r\n", loSetpoint);
    pc.printf("\r\n");
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void openMenu_SetHiSetpoint(void) {
    menuSystem.open(menuSetHiSetpoint); // activate the menu
    pc.printf("---------------------------------------------------\r\n");
    pc.printf("High setpoint menu:\r\n");
    pc.printf("Press 'Mode' to open 'Low setpoint' menu.\r\n");
    pc.printf("Press 'More' to increment high setpoint.\r\n");
    pc.printf("Press 'Less' to decrement high setpoint.\r\n");
    pc.printf("Press 'Save' to save the setoint.\r\n");
    pc.printf("\r\n");

    p_setpoint = &hiSetpoint;
    setpoint = hiSetpoint;
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void openMenu_SetLoSetpoint(void) {
    menuSystem.open(menuSetLoSetpoint); // activate the menu
    pc.printf("---------------------------------------------------\r\n");
    pc.printf("Low setpoint menu:\r\n");
    pc.printf("Press 'Mode' to open 'Main' menu.\r\n");
    pc.printf("Press 'More' to increment low setpoint.\r\n");
    pc.printf("Press 'Less' to decrement low setpoint.\r\n");
    pc.printf("Press 'Save' to save the setpoint.\r\n");
    pc.printf("\r\n");

    p_setpoint = &loSetpoint;
    setpoint = loSetpoint;
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void incrementSetpoint(void) {
    setpoint++;
    pc.printf("Setpoint incremented\r\n");
    pc.printf("setpoint = %i\r\n", setpoint);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void decrementSetpoint(void) {
    setpoint--;
    pc.printf("Setpoint decremented\r\n");
    pc.printf("setpoint = %i\r\n", setpoint);
}


/**
 * @brief
 * @note
 * @param
 * @retval
 */
void saveSetpoint(void) {
    *p_setpoint = setpoint;
    pc.printf("Setpoint saved\r\n");
    pc.printf("\r\n");
}




