/*
 ******************************************************************************
 * @file    ManchesterUART.h
 * @author  Zoltan Hudak
 * @version
 * @date    2017-Nov-22
 * @brief   Manchester code over UART for mbed
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2017 Zoltan Hudak <hudakz@outlook.com>
 *
 * All rights reserved.

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/*
 * This library implements Manchester code using UART serial connection. Each
 * data byte is encoded into two bytes representing two nibbles of the original
 * data byte. These two bytes are then sent over UART serial link connection.
 * The receiver reconstructs the original data byte from the two bytes received.
 * A start and stop pattern are sent to signify the begin and end of a message.
 *
 * The library is based on the article published by Adrian Mills:
 * http://www.quickbuilder.co.uk/qb/articles/Manchester_encoding_using_RS232.pdf
 */
#ifndef MANCHESTERUART_H
#define MANCHESTERUART_H

#include "mbed.h"
#include "ManchesterMsg.h"

#define START   0xF0    // start pattern

#define STOP    0x0F    // stop pattern
class   ManchesterUART
{
    enum Error
    {
        NO_ERROR,
        ILLEGAL_CODE,
        RX_TIMEOUT,
        BUF_OVERRUN
    };
public:
    /**
     * @brief   Creates a ManchesterUART object
     * @note
     * @param   txPin Pin name of transmitter line
     * @param   rxPin Pin name of receiver line
     * @param   preamble Number of start patterns at the begin of each transmission
     * @param   baudrate Communication bit rate in bits per second. Defaults to 115200bps
     * @param   timeout Receive timeout in seconds. Defaults to 5 seconds.
     * @retval
     */
    ManchesterUART
    (
        PinName txPin,
        PinName rxPin,
        int     baudrate = 9600,    /* bits/sec */
        uint8_t preamble = 8,       /* number of start patterns */
        float   rxTimeout = 5       /* seconds */
    ) :
    _serial(txPin, rxPin, baudrate),
    _rxTimeout(rxTimeout),
    _preamble(preamble),
    _error(NO_ERROR)
    { }

    ~       ManchesterUART(void)    { }
    void    transmit(ManchesterMsg& msg);
    void    attach(Callback<void ()> func, SerialBase::IrqType type = SerialBase::RxIrq)    { _serial.attach(func, type); }
    bool    receive(ManchesterMsg& msg);
    bool    readable()                      { return _serial.readable(); }
    void    baud(int baudrate)              { _serial.baud(baudrate); }
    void    setPreamble(uint8_t length)     { _preamble = length; }
    void    setRxTimeout(int seconds)       { _rxTimeout = seconds; }
    Error   lastError()                     { return _error; }
private:
    Serial  _serial;
    char*   _data;      // data array
    size_t  _len;       // data length in bytes
    size_t  _maxLen;    // data maximum length
    Timeout _timeout;   // timeout
    float   _rxTimeout; // timeout time in seconds
    uint8_t _preamble;  // number of start patterns
    Error   _error;     // error flag
    void    rxTimeout(void);
    void    transmitByte(uint8_t data);
    uint8_t receiveByte(void);
    uint8_t getNibble(uint8_t nibble);
};
#endif // MANCHESTERUART_H
