/*
 ******************************************************************************
 * @file    ManchesterMsg.h
 * @author  Zoltan Hudak
 * @version
 * @date    2017-May-16
 * @brief   Message container
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2017 Zoltan Hudak <hudakz@outlook.com>
 *
 * All rights reserved.

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef MANCHESTERMSG_H
#define MANCHESTERMSG_H

//#define DEBUG 1

/** ManchesterMsg class
 */
class   ManchesterMsg
{
    size_t  _max;   // Max length (capacity) of data field in bytes

public:
    char*   data;   // Data field
    size_t  len;    // Length of data in bytes

    /** Creates empty message of specified capacity.
     */
    ManchesterMsg(size_t max) : _max(max), data(new char[max]) { len = 0; }

    ~ ManchesterMsg(void)   { delete[] data; }

    /** Copy constructor.
     */
    ManchesterMsg(const ManchesterMsg& msg) { len = msg.len; memcpy(data, msg.data, msg.len); }

    /** Returns message maximum length (capacity)
     */
    size_t  maxLen(void)    { return _max; }

    /** Clears message content
     */
    void    clear(void)     { len = 0; memset(data, 0, _max); }

    /** Inserter operator: Appends data (value) to message
     */
    template<class T>
    ManchesterMsg &operator<<(const T val) {
        if(len + sizeof(T) <= _max) {
            memcpy(&data[len], &val, sizeof(T));
            len += sizeof(T);
        }

#if DEBUG
        else {
            printf("Error: Cannot append data. Exceeding max data length!\r\n");
        }
#endif
        return *this;
    }

    /** Inserter operator: Appends array of char to message
     */
    ManchesterMsg &operator<<(const char* str) {
        size_t  strLen = strlen(str);
        if(len + strLen + 1 <= _max) {
            memcpy(data + len, (char*)str, strLen);
            len += strLen;
            data[len++] = '\0';
        }

#if DEBUG
        else {
            printf("Error: Cannot append data. Exceeding max data length!\r\n");
        }
#endif
        return *this;
    }

    /** Extractor operator: Extracts data (value) from message
     */
    template<class T>
    ManchesterMsg &operator>>(T& val) {
        if(sizeof(T) <= len) {
            val = *reinterpret_cast < T * > (&data[0]);
            len -= sizeof(T);
            memcpy(data, data + sizeof(T), len);
        }

#if DEBUG
        else {
            printf("Error: Cannot extract data. Exceeding data length!\r\n");
        }
#endif
        return *this;
    }

    /** Extractor operator: Extracts array of char from message
     */
    ManchesterMsg &operator>>(char* str) {
        size_t  strLen = strlen(data);
        if(strLen <= len) {
            memcpy(str, data, strLen + 1);
            len -= (strLen + 1);
            memcpy(data, data + strLen + 1, len);
        }

#if DEBUG
        else {
            printf("Error: Cannot extract data. Exceeding data length!\r\n");
        }
#endif
        return *this;
    }
};
#endif // MANCHESTERMSG_H

