/*
 ******************************************************************************
 * @file    Manchester.h
 * @author  Zoltan Hudak
 * @version
 * @date    2017-May-16
 * @brief   Manchester code for mbed
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2017 Zoltan Hudak <hudakz@outlook.com>
 *
 * All rights reserved.

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/*
   This library implements Manchester code according to both IEEE 802.3 
   and G.E. Thomas' conventions. 
   • A '0' is expressed by a high-to-low transition, a '1' by low-to-high transition 
     in the IEEE 802.3 convention. The reverse is true in the G.E. Thomas' convention.
   • The transitions which signify '0' or '1' occur at the midpoint of a period. 
   • Transitions at the start of a period are overhead and don't signify data. 
   • Least significant bit is sent first 
   • There is one synchronization pulse at the begin of transmission 
   
   The IEEE 802.3 convention is used by default.
   Select a convention to be used by commenting or uncommenting 
   the line "#define G_E_THOMAS 1" below. 
*/
#ifndef MANCHESTER_H
#define MANCHESTER_H

#include "mbed.h"
#include "ManchesterMsg.h"

//Uncomment the following line to use G.E.Thomas' convention

//#define G_E_THOMAS 1

class   Manchester
{
    enum State
    {
        IDLE,
        SYNCH_START,
        SYNCH_NEXT,
        SYNCH_END,
        SETUP,
        TRANSITION,
        COMPLETE,
        LISTEN,
        DECODE_START,
        DECODE,
        STOP,
        ERROR
    };
public:
    Manchester
    (
        PinName     txPin,          /* transmitter pin name */
        PinName     rxPin,          /* receiver pin name */
        uint32_t    speed = 1200,   /* speed in bits per second */
        uint8_t     tol = 25,       /* pulse width tolerance (+/-) in % */
        float       rxTimeout = 5   /* receive timeout */
    );
    ~       Manchester(void)    { }
    void    transmit(ManchesterMsg& msg);
    bool    receive(ManchesterMsg& msg);
    void    setPreamble(uint8_t preamble)   { _preamble = preamble; }
    void    setRxTimeout(float seconds)     { _rxTimeout = seconds; }
    char*   lastError()                     { return _error; }
private:
    DigitalOut  _tx;                // transmitter line
    InterruptIn _rx;                // receiver line
    Ticker      _txTicker;          // transmitter ticker
    Timeout     _timeout;           // timeout (watchdog)
    Timer       _timer;
    uint32_t    _midBitTime;        // mid-bit time [us]
    uint32_t    _minPulseWidth;     // minimum pulse width [us]
    uint32_t    _maxPulseWidth;     // maximum pulse width [us]
    uint32_t    _tol;
    float       _rxTimeout;         // reception timeout [s]
    State       _state;             // state
    char*       _data;              // data array
    size_t      _len;               // data length in bytes
    size_t      _maxLen;            // maximum length
    size_t      _preamble;          // number of start patterns
    char*       _error;
    void        transmission(void);
    void        reception(void);
    void        onTxTimeout(void);
    void        onRxTimeout(void);
};
#endif // MANCHESTER_H
