#include "mbed.h"

#include "EthernetInterface.h"
#include "MQTTNetwork.h"
#include "MQTTmbed.h"
#include "MQTTClient.h"

#define IP          "192.168.1.35"
#define GATEWAY     "192.168.1.1"
#define NETMASK     "255.255.255.0"
#define BROKER_IP   "192.168.1.31"  // IP address of your MQTT broker. Change to match your case.
#define MQTT_PORT   1883            // MQTT port

const uint8_t                           MAC[6] = { 0, 1, 2, 3, 4, 5 };
const char                              payload[] = "Hello, World!";

// Global variables
char                                    topic[256];
EthernetInterface                       net;
MQTTNetwork                             mqttNetwork(&net);
MQTT::Client<MQTTNetwork, Countdown>    mqttClient(mqttNetwork);
Timer                                   timer;

// Function prototypes
void                                    onMqttMsgReceived(MQTT::MessageData& md);
void                                    publishMsg();

/**
 * @brief
 * @note
 * @param
 * @retval
 */
FileHandle* mbed::mbed_override_console(int)
{
    static BufferedSerial   myConsole(USBTX, USBRX, 115200);
    return &myConsole;
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
int main(void)
{
    // Bring up the ethernet interface
    net.set_network(IP, NETMASK, GATEWAY);  // include this to use static IP address
    net.get_emac().set_hwaddr(MAC);         // set MAC address
    printf("MQTT client example\n");
    if (net.connect() != 0) {
        printf("Error connecting\n");
        return -1;
    }

    // Show network address
    SocketAddress addr;
    net.get_ip_address(&addr);
    printf("IP address: %s\n", addr.get_ip_address() ? addr.get_ip_address() : "None");
    net.get_netmask(&addr);
    printf("Netmask: %s\n", addr.get_ip_address() ? addr.get_ip_address() : "None");
    net.get_gateway(&addr);
    printf("Gateway: %s\n", addr.get_ip_address() ? addr.get_ip_address() : "None");

    printf("Connecting to %s:%d\r\n", BROKER_IP, MQTT_PORT);
    nsapi_error_t error = mqttNetwork.connect(BROKER_IP, MQTT_PORT);
    if (error == NSAPI_ERROR_OK) {
        printf("Server connected.\r\n");
    }
    else {
        printf("Cannot connect server.\r\n");
        return -1;
    }

    MQTTPacket_connectData data = MQTTPacket_connectData_initializer;
    data.MQTTVersion = 3;
    data.clientID.cstring = "MyMqttClient";
    data.username.cstring = NULL;
    data.password.cstring = NULL;

    // Connect MQTT client to MQTT broker
    printf("Connecting MQTT Broker\r\n");
    if (mqttClient.connect(data) != 0) {
        printf("Cannot connect MQTT broker.\r\n");
        return -1;
    };
    printf("MQTT Broker connected.\r\n");

    // Subscribe to topics
    mqttClient.subscribe("outdoorTemperature", MQTT::QOS0, onMqttMsgReceived);

    // Start timer
    timer.start();
    
    // Main thread loop
    while (1) {
        mqttClient.yield(10);
        if (timer.read_ms() > 1000) {
            timer.reset();
            publishMsg();   // once a second publish the MQTT message
        }
    }
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void onMqttMsgReceived(MQTT::MessageData& md)
{
    memset(topic, 0, sizeof(topic));
    memcpy(topic, md.topicName.lenstring.data, md.topicName.lenstring.len);
    printf("topic: %s\r\n", topic);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void publishMsg()
{
    MQTT::Message mqttMsg;

    mqttMsg.qos = MQTT::QOS0;
    mqttMsg.retained = false;
    mqttMsg.dup = false;
    mqttMsg.payload = (void*)payload;
    mqttMsg.payloadlen = strlen(payload);    
    mqttClient.publish("topic_hello_world", mqttMsg);
}