// In this project an MQTT client is created.
// It is publishing parameters measured by a DHT11 sensor.
// Ethernet connection is assured via an ENC28J60 module.

#include "mbed.h"
#include "UIPEthernet.h"
#include "MQTTClient.h"
#include "DHT11.h"

Serial          pc(USBTX, USBRX);

// MAC address must be unique within the connected network. Modify as appropriate.
const uint8_t   MY_MAC[6] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05 };

#define DHCP    1   // comment out this line if you'd like to use static IP address

#if !defined(DHCP)
// In case you'd like to use static IP address:
// IP address must be unique and compatible with your network.
// Change as appropriate.
const IPAddress MY_IP(192, 168, 1, 181);
#endif

// uIPEthernet is the name of a global instance of UIPEthernet.
// Do not change the name! It is used within the UIPEthernet library.
// Adapt the SPI pin names to your mbed platform/board if not present yet.

#if defined(TARGET_LPC1768)
UIPEthernet  uIPEthernet(p11, p12, p13, p8);         // mosi, miso, sck, cs
DHT11        dht11(p6);
#elif defined(TARGET_NUCLEO_F103RB) || defined(TARGET_NUCLEO_L152RE) || defined(TARGET_NUCLEO_F030R8)  \
   || defined(TARGET_NUCLEO_F401RE) || defined(TARGET_NUCLEO_F302R8) || defined(TARGET_NUCLEO_L053R8)  \
   || defined(TARGET_NUCLEO_F411RE) || defined(TARGET_NUCLEO_F334R8) || defined(TARGET_NUCLEO_F072RB)  \
   || defined(TARGET_NUCLEO_F091RC) || defined(TARGET_NUCLEO_F303RE) || defined(TARGET_NUCLEO_F070RB)  \
   || defined(TARGET_KL25Z ) || defined(TARGET_KL46Z) || defined(TARGET_K64F) || defined(TARGET_KL05Z) \
   || defined(TARGET_K20D50M) || defined(TARGET_K22F) \
   || defined(TARGET_NRF51822) \
   || defined(TARGET_RZ_A1H)
UIPEthernet  uIPEthernet(D11, D12, D13, D10);        // mosi, miso, sck, cs
DHT11        dht11(PC_14);
#endif

char  message_buff[100];
void  onMessage(char* topic, uint8_t* payload, unsigned int length);

// MQTT broker is like a post office.
// Its task is to distribute messages published by clients to all subscribers (other clients).
// So messages published by this client will be sent to the broker.
// Then the broker will send them to all clients which subscribed to such topics.
// Also this client will receive all messages with topics it subscribed to.
// 'Mosquitto' is a free implementation of MQTT broker for Linux machines (e.g. Raspberry Pi, Ubuntu etc.)
IPAddress       serverIP(192, 168, 1, 30);  // IP address of your MQTT broker (adapt)
EthernetClient  ethernetClient;
void            onMqttMessage(char* topic, uint8_t* payload, unsigned int length);
MQTTClient      mqttClient(serverIP, 1883, onMqttMessage, ethernetClient);
const int       INTERVAL = 10;  // Interval for publishing messages (in seconds)


/**
 * @brief   Main
 * @note
 * @param
 * @retval
 */
int main(void) {
    const int   MAX_COUNT = 5;
    int         i = 0;
    bool        connected = false;
    time_t      t = 0;
    time_t      lastTime = 0;

    // initialize the ethernet device
    
#if defined(DHCP)
    pc.printf("Searching for DHCP server..\r\n");
 
    if (uIPEthernet.begin(MY_MAC) != 1) {
        pc.printf("No DHCP server found.\r\n");
        pc.printf("Exiting application.\r\n");
        return 0;
    }
    pc.printf("DHCP server found and configuration info received.\r\n");
#else
    uIPEthernet.begin(MY_MAC, MY_IP);
#endif
    pc.printf("Local IP = %s\r\n", uIPEthernet.localIP().toString());       
    pc.printf("Connecting to MQTT broker ..\r\n");
    do {
        wait(1.0);
        connected = mqttClient.connect("myMqttClient");
    } while(!connected && (i < MAX_COUNT));

    if (connected) {
        pc.printf("MQTT broker connected.\r\n");
        // The client can subscribe to as many MQTT messages as you like.
        mqttClient.subscribe("example/hello"); 
        mqttClient.subscribe("boiler/outlet/temperature");
        pc.printf("Subscribed to:\r\n");
        pc.printf("    example/hello\r\n");
        pc.printf("    boiler/outlet/temperature\r\n");
    }
    else {
        pc.printf("Failed to connect to MQTT broker.\r\n");
    }

    while (1) {
        t = time(NULL);
        if(t >= (lastTime + INTERVAL)) {
            lastTime = t;
            if(connected) {
                pc.printf("---------------------\r\n");
                pc.printf("%ds:\r\n", t);

                int state = dht11.readData();
                if(state == DHT11::OK) {
                    float   hum = dht11.readHumidity();
                    sprintf(message_buff, "%4.1f", hum);
                    pc.printf("  hum = %s%%\r\n", message_buff);
                    mqttClient.publish("outdoor/humidity", message_buff);

                    float   temp = dht11.readTemperature();
                    sprintf(message_buff, "%5.1f", temp);
                    pc.printf("  temp = %s'C\r\n", message_buff);
                    mqttClient.publish("outdoor/temperature", message_buff);

                    float   dewPoint = temp - (100 - hum) / 5.0;
                    sprintf(message_buff, "%5.1f", dewPoint);
                    pc.printf("  dew point = %s'C\r\n", message_buff);
                    mqttClient.publish("outdoor/dewpoint", message_buff);
                }
                else
                    pc.printf("  DHT11 error: %d\r\n", state);
            }
        }

        mqttClient.loop();  // MQTT client loop processing
    }
}

/**
 * @brief   Called on new MQTT message arrival
 * @note    
 * @param   topic:      The topic of the new message
 *          payload:    The payload of the new message
 *          length:     Payload's length
 * @retval
 */
void onMqttMessage(char* topic, uint8_t* payload, unsigned int length) {
    int i = 0;

    pc.printf("Message arrived:\r\n");
    pc.printf("    Topic: %s\r\n", topic);
    pc.printf("    Length: %d\r\n", length);

    // create character buffer with ending null terminator (string)
    for (i = 0; i < length; i++) {
        message_buff[i] = payload[i];
    }

    message_buff[i] = '\0';
    pc.printf("    Payload: %s\r\n", message_buff);
}

