#include "FSMC8.h"
#include "stdint.h"

// Addres bit A18 is used to select LCD's REGISTER or RAM: A18 = 0b00000000000001000000000000000000 = 0x00040000
volatile unsigned char* lcd_reg = (unsigned char*)0x60000000;   // FSMC BANK1, A18 = 0 -> write to LCD's REGISTER
volatile unsigned char* lcd_ram = (unsigned char*)0x60040000;   // FSMC BANK1, A18 = 1 -> read/write to LCD's RAM

/**
 * @brief
 * @note
 * @param
 * @retval
 */
FSMC8::FSMC8(PinName reset) :
    _reset(reset)
{
    _reset = 1;
}


/**
 * @brief
 * @note
 * @param
 * @retval
 */
void FSMC8::wr_cmd8(unsigned char cmd)
{
    lcd_reg[0] = cmd;   // Write to LCD's register
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void FSMC8::wr_data8(unsigned char data)
{
    lcd_ram[0] = data;  // Write to LCD's RAM
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void FSMC8::wr_cmd16(unsigned short cmd)
{
    wr_cmd8(cmd >> 8);      // Write MSB to LCD's register
    wr_cmd8(cmd & 0xFF);    // Write LSB to LCD's register
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void FSMC8::wr_data16(unsigned short data)
{
    wr_data8(data >> 8);    // Write MSB to LCD's RAM
    wr_data8(data & 0xFF);  // Write LSB to LCD's RAM

}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void FSMC8::wr_gram(unsigned short data)
{
    wr_data16(data);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void FSMC8::wr_gram(unsigned short data, unsigned int count)
{
    if ((data >> 8) == (data & 0xFF)) {
        count <<= 1;
        while (count) {
            wr_data8(data);
            count--;
        }
    }
    else {
        while (count) {
            wr_data16(data);
            count--;
        }
    }
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void FSMC8::wr_grambuf(unsigned short* data, unsigned int lenght)
{
    while (lenght) {
        wr_data16(*data++);
        lenght--;
    }
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
unsigned char FSMC8::rd_data8(unsigned int i /* = 0 */)
{
    return lcd_ram[i];  // Read from LCD's RAM
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
unsigned short FSMC8::rd_gram(bool convert)
{
    unsigned int    r = 0;

    rd_data8(); // dummy read
    r |= rd_data8();
    r <<= 8;
    r |= rd_data8();

    if (convert) {
        r <<= 8;
        r |= rd_data8();

        // gram is 18bit/pixel, if you set 16bit/pixel (cmd 3A), during writing the 16bits are expanded to 18bit
        // during reading, you read the raw 18bit gram
        r = RGB24to16((r & 0xFF0000) >> 16, (r & 0xFF00) >> 8, r & 0xFF);   // 18bit pixel padded to 24bits, rrrrrr00_gggggg00_bbbbbb00, converted to 16bit
    }

    //    _bus.output();
    return (unsigned short)r;
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
unsigned int FSMC8::rd_reg_data32(unsigned char reg)
{
    unsigned char  data[4] = { 0 };
    unsigned int*  r = (unsigned int*)&data[0];

    wr_cmd8(reg);
    data[3] = rd_data8(0);
    data[2] = rd_data8(0);
    data[1] = rd_data8(0);
    data[0] = rd_data8(0);

    return *r;
}

// in Par mode EXTC regs (0xB0-0xFF) can be directly read
unsigned int FSMC8::rd_extcreg_data32(unsigned char reg, unsigned char SPIreadenablecmd)
{
    return rd_reg_data32(reg);
}

// ILI932x specific
void FSMC8::dummyread()
{
    rd_data8(); // dummy read
}

// ILI932x specific
void FSMC8::reg_select(unsigned char reg, bool forread)
{
    wr_cmd8(0);     // write MSB
    wr_cmd8(reg);   // write LSB
}

// ILI932x specific
void FSMC8::reg_write(unsigned char reg, unsigned short data)
{
    wr_cmd8(0);     // write MSB
    wr_cmd8(reg);   // write LSB
    wr_data16(data);
}

// ILI932x specific
unsigned short FSMC8::reg_read(unsigned char reg)
{
    unsigned short  r = 0;

    wr_cmd8(0);
    wr_cmd8(reg);
    r |= rd_data8() & 0xFF;
    r <<= 8;
    r |= rd_data8() & 0xFF;

    return r;
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void FSMC8::hw_reset()
{
    _reset = 1;
    wait_ms(15);
    _reset = 0; // display reset
    wait_ms(2);
    _reset = 1; // end reset
    wait_ms(100);
}
