/* mbed UniGraphic library - universal TFT driver class
 * Copyright (c) 2015 Giuliano Dianda
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * Derived work of:
 *
 * mbed library for 240*320 pixel display TFT based on ILI9341 LCD Controller
 * Copyright (c) 2013 Peter Drescher - DC2PD
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include "TFT.h"

//#include "mbed_debug.h"

#define MIPI_DCS_REV1   (1<<0)
#define AUTO_READINC    (1<<1)
#define READ_BGR        (1<<2)
#define READ_LOWHIGH    (1<<3)
#define READ_24BITS     (1<<4)
#define XSA_XEA_16BIT   (1<<5)
#define READ_NODUMMY    (1<<6)
#define INVERT_GS       (1<<8)
#define INVERT_SS       (1<<9)
#define MV_AXIS         (1<<10)
#define INVERT_RGB      (1<<11)
#define REV_SCREEN      (1<<12)
#define FLIP_VERT       (1<<13)
#define FLIP_HORIZ      (1<<14)

#define SWAP(a, b) { \
        a ^= b; \
        b ^= a; \
        a ^= b; \
    }

#if DEVICE_PORTINOUT

/**
 * @brief
 * @note
 * @param
 * @retval
 */
TFT::TFT
(
    proto_t         displayproto,
    PortName        port,
    PinName         CS,
    PinName         reset,
    PinName         DC,
    PinName         WR,
    PinName         RD,
    const int       lcdsize_x,
    const int       lcdsize_y,
    const char*     name
) :
    GraphicsDisplay(name),
    screensize_X(lcdsize_x),
    screensize_Y(lcdsize_y)
{
    if (displayproto == PAR_8)
        proto = new PAR8(port, CS, reset, DC, WR, RD);
    else
    if (displayproto == PAR_16)
        proto = new PAR16(port, CS, reset, DC, WR, RD);
    useNOP = false;
    scrollbugfix = 0;
    mipistd = false;
    set_orientation(0);
    foreground(White);
    background(Black);
    set_auto_up(false); //we don't have framebuffer
    topfixedareasize = 0;
    scrollareasize = 0;
    usefastwindow = false;
    fastwindowready = false;
    is18bit = false;
    isBGR = false;

    //  cls();
    //  locate(0,0);
}
#endif

/**
 * @brief
 * @note
 * @param
 * @retval
 */
TFT::TFT
(
    proto_t         displayproto,
    PinName*        buspins,
    PinName         CS,
    PinName         reset,
    PinName         DC,
    PinName         WR,
    PinName         RD,
    const int       lcdsize_x,
    const int       lcdsize_y,
    const char*     name
) :
    GraphicsDisplay(name),
    screensize_X(lcdsize_x),
    screensize_Y(lcdsize_y)
{
    if (displayproto == BUS_8) {
        PinName pins[16];
        for (int i = 0; i < 16; i++)
            pins[i] = NC;
        for (int i = 0; i < 8; i++)
            pins[i] = buspins[i];
        proto = new BUS8(pins, CS, reset, DC, WR, RD);
    }
    else
    if (displayproto == BUS_16) {
        proto = new BUS16(buspins, CS, reset, DC, WR, RD);
    }

    useNOP = false;
    scrollbugfix = 0;
    mipistd = false;
    set_orientation(0);
    foreground(White);
    background(Black);
    set_auto_up(false); //we don't have framebuffer
    topfixedareasize = 0;
    scrollareasize = 0;
    usefastwindow = false;
    fastwindowready = false;
    is18bit = false;
    isBGR = false;

    //  cls();
    //  locate(0,0);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
TFT::TFT
(
    proto_t         displayproto,
    int             Hz,
    PinName         mosi,
    PinName         miso,
    PinName         sclk,
    PinName         CS,
    PinName         reset,
    PinName         DC,
    const int       lcdsize_x,
    const int       lcdsize_y,
    const char*     name
) :
    GraphicsDisplay(name),
    screensize_X(lcdsize_x),
    screensize_Y(lcdsize_y)
{
    if (displayproto == SPI_8) {
        proto = new SPI8(Hz, mosi, miso, sclk, CS, reset, DC);
        useNOP = false;
    }
    else
    if (displayproto == SPI_16) {
        proto = new SPI16(Hz, mosi, miso, sclk, CS, reset, DC);
        useNOP = true;
    }

    scrollbugfix = 0;
    mipistd = false;
    set_orientation(0);
    foreground(White);
    background(Black);
    set_auto_up(false);
    topfixedareasize = 0;
    scrollareasize = 0;
    usefastwindow = false;
    fastwindowready = false;
    is18bit = false;
    isBGR = false;

    //  locate(0,0);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
TFT::TFT(proto_t displayproto, PinName reset, const int lcdsize_x, const int lcdsize_y, const char* name) :
    GraphicsDisplay(name),
    screensize_X(lcdsize_x),
    screensize_Y(lcdsize_y)
{
    if (displayproto == FSMC_8) {
        proto = new FSMC8(reset);
        useNOP = false;
    }

    scrollbugfix = 0;
    mipistd = false;

    //    set_orientation(0);
    foreground(White);
    background(Black);
    set_auto_up(false); //we don't have framebuffer
    topfixedareasize = 0;
    scrollareasize = 0;
    usefastwindow = false;
    fastwindowready = false;
    is18bit = false;
    isBGR = false;

    //  cls();
    //  locate(0,0);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::wr_cmd8(unsigned char cmd)
{
    if (useNOP)
        proto->wr_cmd16(cmd);   // 0x0000|cmd, 00 is NOP cmd for TFT
    else
        proto->wr_cmd8(cmd);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::wr_data8(unsigned char data)
{
    proto->wr_data8(data);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::wr_data16(unsigned short data)
{
    proto->wr_data16(data);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::wr_gram(unsigned short data)
{
    proto->wr_gram(data);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::wr_gram(unsigned short data, unsigned int count)
{
    proto->wr_gram(data, count);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::wr_grambuf(unsigned short* data, unsigned int lenght)
{
    proto->wr_grambuf(data, lenght);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
unsigned short TFT::rd_gram()
{
    return proto->rd_gram(is18bit); // protocol will handle 18to16 bit conversion
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
unsigned int TFT::rd_reg_data32(unsigned char reg)
{
    return proto->rd_reg_data32(reg);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
unsigned int TFT::rd_extcreg_data32(unsigned char reg, unsigned char SPIreadenablecmd)
{
    return proto->rd_extcreg_data32(reg, SPIreadenablecmd);
}

//for TFT, just send data, position counters are in hw
void TFT::window_pushpixel(unsigned short color)
{
    proto->wr_gram(color);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::window_pushpixel(unsigned short color, unsigned int count)
{
    proto->wr_gram(color, count);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::window_pushpixelbuf(unsigned short* color, unsigned int lenght)
{
    proto->wr_grambuf(color, lenght);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::hw_reset()
{
    proto->hw_reset();
    BusEnable(true);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::BusEnable(bool enable)
{
    proto->BusEnable(enable);
}

// color TFT can rotate in hw (swap raw<->columns) for landscape views
void TFT::set_orientation(int orient)
{
    //ientation = o;

    //wr_cmd8(0x36);
    //switch (orientation) {
    //    case 0:                 // default, portrait view 0°
    //        if (mipistd)
    //            wr_data8(0x0A); // this is in real a vertical flip enabled, seems most displays are vertical flipped
    //        else
    //            wr_data8(0x48); //for some other ILIxxxx
    //        set_width(screensize_X);
    //        set_height(screensize_Y);
    //        break;
    //    case 1:                 // landscape view +90°
    //        if (mipistd)
    //            wr_data8(0x28);
    //        else
    //            wr_data8(0x29); //for some other ILIxxxx
    //        set_width(screensize_Y);
    //        set_height(screensize_X);
    //        break;
    //    case 2:                 // portrait view +180°
    //        if (mipistd)
    //            wr_data8(0x09);
    //        else
    //            wr_data8(0x99); //for some other ILIxxxx
    //        set_width(screensize_X);
    //        set_height(screensize_Y);
    //        break;
    //    case 3:                 // landscape view -90°
    //        if (mipistd)
    //            wr_data8(0x2B);
    //        else
    //            wr_data8(0xF8); //for some other ILIxxxx
    //        set_width(screensize_Y);
    //        set_height(screensize_X);
    //        break;
    //}

    uint8_t     val;

    orientation = orient & 3;                                       // just perform the operation ourselves on the protected variables
    set_width((orientation & 1) ? screensize_Y : screensize_X);
    set_height((orientation & 1) ? screensize_X : screensize_Y);
    switch (orientation) {
        case 0:                                                     //PORTRAIT:
            val = 0x48;                                             //MY=0, MX=1, MV=0, ML=0, BGR=1
            break;

        case 1:                                                     //LANDSCAPE: 90 degrees
            val = 0x28;                                             //MY=0, MX=0, MV=1, ML=0, BGR=1
            break;

        case 2:                                                     //PORTRAIT_REV: 180 degrees
            val = 0x98;                                             //MY=1, MX=0, MV=0, ML=1, BGR=1
            break;

        case 3:                                                     //LANDSCAPE_REV: 270 degrees
            val = 0xF8;                                             //MY=1, MX=1, MV=1, ML=1, BGR=1
            break;
    }

    bool        _lcd_capable = 1;
    bool        is8347 = 0;
    uint16_t    _lcd_rev = 1;

    uint8_t     _MC;
    uint8_t     _MP;
    uint8_t     _MW;
    uint8_t     _SC;
    uint8_t     _EC;
    uint8_t     _SP;
    uint8_t     _EP;

    uint16_t    GS;
    uint16_t    SS_v;
    uint16_t    ORG;
    uint16_t    REV = _lcd_rev;
    uint8_t     d[3];
    uint16_t    _lcd_ID = tftID;
    uint8_t     _lcd_madctl;

    if (_lcd_capable & INVERT_GS)
        val ^= 0x80;
    if (_lcd_capable & INVERT_SS)
        val ^= 0x40;
    if (_lcd_capable & INVERT_RGB)
        val ^= 0x08;
    if (_lcd_capable & MIPI_DCS_REV1) {
        if (_lcd_ID == 0x6814) {

            //.kbv my weird 0x9486 might be 68140
            GS = (val & 0x80) ? (1 << 6) : 0;                       //MY
            SS_v = (val & 0x40) ? (1 << 5) : 0;                     //MX
            val &= 0x28;                                            //keep MV, BGR, MY=0, MX=0, ML=0
            d[0] = 0;
            d[1] = GS | SS_v | 0x02;                                //MY, MX
            d[2] = 0x3B;
            //WriteCmdParamN(0xB6, 3, d);
            wr_cmd8(0xB6);
            for (int i = 0; i < 3; i++) {
                wr_data8(d[i]);
            }
            goto common_MC;
        }
        else
        if (_lcd_ID == 0x1963 || _lcd_ID == 0x9481 || _lcd_ID == 0x1511) {
            if (val & 0x80)
                val |= 0x01;                                        //GS
            if ((val & 0x40))
                val |= 0x02;                                        //SS
            if (_lcd_ID == 0x1963)
                val &= ~0xC0;
            if (_lcd_ID == 0x9481)
                val &= ~0xD0;
            if (_lcd_ID == 0x1511) {
                val &= ~0x10;                                       //remove ML
                val |= 0xC0;                                        //force penguin 180 rotation
            }

            //val &= (_lcd_ID == 0x1963) ? ~0xC0 : ~0xD0; //MY=0, MX=0 with ML=0 for ILI9481
            goto common_MC;
        }
        else
        if (is8347) {
            uint8_t _MC = 0x02, _MP = 0x06, _MW = 0x22, _SC = 0x02, _EC = 0x04, _SP = 0x06, _EP = 0x08;
            if (_lcd_ID == 0x0065) {

                //HX8352-B
                if (!(val & 0x10))
                    val ^= 0x81;                                    //(!ML) flip MY, GS
                if (orient & 1)
                    _MC = 0x82, _MP = 0x80;
                else
                    _MC = 0x80, _MP = 0x82;
            }

            if (_lcd_ID == 0x5252) {
                val |= 0x02;                                        //VERT_SCROLLON
                if (val & 0x10)
                    val |= 0x04;                                    //if (ML) SS=1 kludge mirror in XXX_REV modes
            }

            goto common_BGR;
        }

common_MC:
        _MC = 0x2A, _MP = 0x2B, _MW = 0x2C, _SC = 0x2A, _EC = 0x2A, _SP = 0x2B, _EP = 0x2B;
common_BGR:
        //WriteCmdParamN(is8347 ? 0x16 : 0x36, 1, &val);
        wr_cmd8(is8347 ? 0x16 : 0x36);
        wr_data8(val);

        _lcd_madctl = val;

        //if (_lcd_ID	== 0x1963) WriteCmdParamN(0x13, 0, NULL);   //NORMAL mode
    }

    // cope with 9320 variants
    else {
        switch (_lcd_ID) {
#define SUPPORT_9225
    #if defined(SUPPORT_9225)

            case 0x9225:
            case 0x9226:
                _SC = 0x37, _EC = 0x36, _SP = 0x39, _EP = 0x38;
                _MC = 0x20, _MP = 0x21, _MW = 0x22;
                GS = (val & 0x80) ? (1 << 9) : 0;
                SS_v = (val & 0x40) ? (1 << 8) : 0;
                //WriteCmdData(0x01, GS | SS_v | 0x001C);             // set Driver Output Control
                wr_cmd8(0x01);
                wr_data8(GS | SS_v | 0x001C);
                goto common_ORG;
    #endif
#define SUPPORT_0139
    #if defined(SUPPORT_0139) || defined(SUPPORT_0154)
    #ifdef SUPPORT_0139

            case 0x0139:
                _SC = 0x46, _EC = 0x46, _SP = 0x48, _EP = 0x47;
                goto common_S6D;
    #endif
#define SUPPORT_0154
    #ifdef SUPPORT_0154

            case 0x0154:
                _SC = 0x37, _EC = 0x36, _SP = 0x39, _EP = 0x38;
                goto common_S6D;
    #endif
                common_S6D : _MC = 0x20, _MP = 0x21, _MW = 0x22;
                GS = (val & 0x80) ? (1 << 9) : 0;
                SS_v = (val & 0x40) ? (1 << 8) : 0;

                // S6D0139 requires NL = 0x27,  S6D0154 NL = 0x28
                //WriteCmdData(0x01, GS | SS_v | ((_lcd_ID == 0x0139) ? 0x27 : 0x28));
                wr_cmd8(0x01);
                wr_data8(GS | SS_v | ((_lcd_ID == 0x0139) ? 0x27 : 0x28));
                goto common_ORG;
    #endif

            case 0x5420:
            case 0x7793:
            case 0x9326:
            case 0xB509:
                _MC = 0x200, _MP = 0x201, _MW = 0x202, _SC = 0x210, _EC = 0x211, _SP = 0x212, _EP = 0x213;
                GS = (val & 0x80) ? (1 << 15) : 0;

                uint16_t    NL;
                NL = ((432 / 8) - 1) << 9;
                if (_lcd_ID == 0x9326 || _lcd_ID == 0x5420)
                    NL >>= 1;
                //WriteCmdData(0x400, GS | NL);
                wr_cmd8(0x400);
                wr_data8(GS | NL);
                goto common_SS;

            default:
                _MC = 0x20, _MP = 0x21, _MW = 0x22, _SC = 0x50, _EC = 0x51, _SP = 0x52, _EP = 0x53;
                GS = (val & 0x80) ? (1 << 15) : 0;
                //WriteCmdData(0x60, GS | 0x2700);                    // Gate Scan Line (0xA700)
                wr_cmd8(0x60);
                wr_data8(GS | 0x2700);
    common_SS:
                SS_v = (val & 0x40) ? (1 << 8) : 0;
                //WriteCmdData(0x01, SS_v);                           // set Driver Output Control
                wr_cmd8(0x01);
                wr_data8(SS_v);
    common_ORG:
                ORG = (val & 0x20) ? (1 << 3) : 0;
    #ifdef SUPPORT_8230
                if (_lcd_ID == 0x8230) {

                    // UC8230 has strange BGR and READ_BGR behaviour
                    if (rotation == 1 || rotation == 2) {
                        val ^= 0x08;                                // change BGR bit for LANDSCAPE and PORTRAIT_REV
                    }
                }
    #endif
                if (val & 0x08)
                    ORG |= 0x1000;                                  //BGR
                _lcd_madctl = ORG | 0x0030;
                //WriteCmdData(0x03, _lcd_madctl);                    // set GRAM write direction and BGR=1.
                wr_cmd8(0x03);
                wr_data8(_lcd_madctl);

                break;
    #ifdef SUPPORT_1289

            case 0x1289:
                _MC = 0x4E, _MP = 0x4F, _MW = 0x22, _SC = 0x44, _EC = 0x44, _SP = 0x45, _EP = 0x46;
                if (rotation & 1)
                    val ^= 0xD0;                                    // exchange Landscape modes
                GS = (val & 0x80) ? (1 << 14) | (1 << 12) : 0;      //called TB (top-bottom)
                SS_v = (val & 0x40) ? (1 << 9) : 0;                 //called RL (right-left)
                ORG = (val & 0x20) ? (1 << 3) : 0;                  //called AM
                _lcd_drivOut = GS | SS_v | (REV << 13) | 0x013F;    //REV=0, BGR=0, MUX=319
                if (val & 0x08)
                    _lcd_drivOut |= 0x0800;                         //BGR
                WriteCmdData(0x01, _lcd_drivOut);                   // set Driver Output Control
                WriteCmdData(0x11, ORG | 0x6070);                   // set GRAM write direction.
                break;
    #endif
        }
    }

    if ((orient & 1) && ((_lcd_capable & MV_AXIS) == 0)) {
        uint16_t    x;
        x = _MC, _MC = _MP, _MP = x;
        x = _SC, _SC = _SP, _SP = x;                                //.kbv check 0139
        x = _EC, _EC = _EP, _EP = x;                                //.kbv check 0139
    }

//    setAddrWindow(0, 0, width() - 1, height() - 1);
//    vertScroll(0, HEIGHT, 0);                                       //reset scrolling after a rotation
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::invert(unsigned char o)
{
    if (o == 0)
        wr_cmd8(0x20);
    else
        wr_cmd8(0x21);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::FastWindow(bool enable)
{
    usefastwindow = enable;
}

// TFT have both column and raw autoincrement inside a window, with internal counters
void TFT::window(int x, int y, int w, int h)
{
    fastwindowready = false;    // end raw/column going to be set to lower value than bottom-right corner
    wr_cmd8(0x2A);
    wr_data16(x);               //start column
    wr_data16(x + w - 1);       //end column
    wr_cmd8(0x2B);
    wr_data16(y);               //start page
    wr_data16(y + h - 1);       //end page
    wr_cmd8(0x2C);              //write mem, just send pixels color next
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::window4read(int x, int y, int w, int h)
{
    fastwindowready = false;
    wr_cmd8(0x2A);
    wr_data16(x);           //start column
    wr_data16(x + w - 1);   //end column
    wr_cmd8(0x2B);
    wr_data16(y);           //start page
    wr_data16(y + h - 1);   //end page
    wr_cmd8(0x2E);          //read mem, just pixelread next
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::pixel(int x, int y, unsigned short color)
{
    if (usefastwindow) {

        //ili9486 does not like truncated 2A/2B cmds, at least in par mode
        if (fastwindowready) {

            //setting only start column/page does speedup, but needs end raw/column previously set to bottom-right corner
            wr_cmd8(0x2A);
            wr_data16(x);                               //start column only
            wr_cmd8(0x2B);
            wr_data16(y);                               //start page only
            wr_cmd8(0x2C);                              //write mem, just send pixels color next
        }
        else {
            window(x, y, width() - x, height() - y);    // set also end raw/column to bottom-right corner
            fastwindowready = true;
        }
    }
    else
        window(x, y, 1, 1);

    //  proto->wr_gram(color);   // 2C expects 16bit parameters
    wr_gram(color);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
unsigned short TFT::pixelread(int x, int y)
{
    if (usefastwindow) {

        //ili9486 does not like truncated 2A/2B cmds, at least in par mode
        if (fastwindowready) {

            //setting only start column/page does speedup, but needs end raw/column previously set to bottom-right corner
            wr_cmd8(0x2A);
            wr_data16(x);                                   //start column only
            wr_cmd8(0x2B);
            wr_data16(y);                                   //start page only
            wr_cmd8(0x2E);                                  //read mem, just pixelread next
        }
        else {
            window4read(x, y, width() - x, height() - y);   // set also end raw/column to bottom-right corner
            fastwindowready = true;
        }
    }
    else
        window4read(x, y, 1, 1);

    unsigned short  color;
    //  proto->wr_gram(color);   // 2C expects 16bit parameters

    color = rd_gram();
    if (isBGR)
        color = BGR2RGB(color);                             // in case, convert BGR to RGB (should depend on cmd36 bit3) but maybe is device specific
    return color;
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::setscrollarea(int startY, int areasize)   // ie 0,480 for whole screen
{
    unsigned int    bfa;
    topfixedareasize = startY;
    scrollareasize = areasize;
    wr_cmd8(0x33);
    wr_data16(topfixedareasize);                //num lines of top fixed area
    wr_data16(scrollareasize + scrollbugfix);   //num lines of vertical scroll area, +1 for ILI9481 fix
    if ((areasize + startY) > screensize_Y)
        bfa = 0;
    else
        bfa = screensize_Y - (areasize + startY);
    wr_data16(bfa);                             //num lines of bottom fixed area
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::scroll(int lines) // ie 1= scrollup 1, 479= scrolldown 1
{
    wr_cmd8(0x37);
    wr_data16(topfixedareasize + (lines % scrollareasize)); // select the (absolute)line which will be displayed as first scrollarea line
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::scrollreset()
{
    wr_cmd8(0x13);  //normal display mode
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void TFT::cls(void)
{
    WindowMax();

    //  proto->wr_gram(_background,screensize_X*screensize_Y);
    //  proto->wr_gram(0,screensize_X*screensize_Y);
    wr_gram(_background, screensize_X * screensize_Y);
}

// try to get read gram pixel format, could be 16bit or 18bit, RGB or BGR
void TFT::auto_gram_read_format()
{
    unsigned short  px = 0xCDB1;
    unsigned short  rback, rback18;
    pixel(0, 0, px);
    window4read(0, 0, 1, 1);
    rback = proto->rd_gram(0);      // try 16bit
    window4read(0, 0, 1, 1);
    rback18 = proto->rd_gram(1);    // try 18bit converted to 16
    if ((rback18 == px) || (BGR2RGB(rback18) == px)) {
        is18bit = true;
        if (BGR2RGB(rback18) == px)
            isBGR = true;
    }
    else
    if ((rback == px) || (BGR2RGB(rback) == px)) {
        if (BGR2RGB(rback) == px)
            isBGR = true;
    }

    //   debug("\r\nIdentify gram read color format,\r\nsent %.4X read16 %.4X(bgr%.4X) read18 %.4X(bgr%.4X)", px, rback, BGR2RGB(rback), rback18, BGR2RGB(rback18));
}

// try to identify display controller
void TFT::identify()
{
    // MIPI std read ID cmd

    tftID = rd_reg_data32(0xBF);
    mipistd = true;

    //   debug("ID MIPI : 0x%8X\r\n",tftID);
    if (((tftID & 0xFF) == ((tftID >> 8) & 0xFF)) && ((tftID & 0xFF) == ((tftID >> 16) & 0xFF))) {
        mipistd = false;

        // ILI specfic read ID cmd
        tftID = rd_reg_data32(0xD3) >> 8;

        //    debug("ID ILI : 0x%8X\r\n",tftID);
    }

    if (((tftID & 0xFF) == ((tftID >> 8) & 0xFF)) && ((tftID & 0xFF) == ((tftID >> 16) & 0xFF))) {

        // ILI specfic read ID cmd with ili9341 specific spi read-in enable 0xD9 cmd
        tftID = rd_extcreg_data32(0xD3, 0xD9);

        //    debug("ID D9 extc ILI : 0x%8X\r\n",tftID);
    }

    if (((tftID & 0xFF) == ((tftID >> 8) & 0xFF)) && ((tftID & 0xFF) == ((tftID >> 16) & 0xFF))) {

        // ILI specfic read ID cmd with ili9486/88 specific spi read-in enable 0xFB cmd
        tftID = rd_extcreg_data32(0xD3, 0xFB);

        //    debug("ID D9 extc ILI : 0x%8X\r\n",tftID);
    }

    if (((tftID & 0xFF) == ((tftID >> 8) & 0xFF)) && ((tftID & 0xFF) == ((tftID >> 16) & 0xFF)))
        tftID = 0xDEAD;
    if ((tftID & 0xFFFF) == 0x9481)
        scrollbugfix = 1;
    else
        scrollbugfix = 0;
    if ((tftID & 0xFFFF) == 0x6814) {
        mipistd = false;    // RM68140
    }

    hw_reset();             // in case wrong cmds messed up important settings
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
int TFT::sizeX()
{
    return screensize_X;
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
int TFT::sizeY()
{
    return screensize_Y;
}
