/*
 * Reading built-in temperature sensor of STM32F103RB chip (on a NUCLEO-F103RB board)
 */

#include "mbed.h"

/* 
 * STM32F103x data-sheet:
 * 5.3.19 Temperature sensor characteristics 
 * Table 50. TS characteristics, Page 80
 */
const float         AVG_SLOPE   = 4.3E-03;      // slope (gradient) of temperature line function  [V/°C]
const float         V25         = 1.43;         // sensor's output voltage at 25°C [V]
const float         ADC_TO_VOLT = 3.3 / 4096;   // conversion coefficient of digital value to voltage [V] 
                                                // when using 3.3V ref. voltage at 12-bit resolution (2^12 = 4096)

Serial              pc(USBTX, USBRX);
DigitalOut          led(PC_13);
ADC_HandleTypeDef   hadc1;                      // ADC handle
uint16_t            adcValue;                   // digital value of sensor
float               vSense;                     // sensor's output voltage [V]
float               temp;                       // sensor's temperature [°C]

/* ADC1 init function */
void MX_ADC1_Init(void)
{
    ADC_ChannelConfTypeDef  sConfig;

    /**Common config
        */
    hadc1.Instance = ADC1;
    hadc1.Init.ScanConvMode = ADC_SCAN_DISABLE;
    hadc1.Init.ContinuousConvMode = DISABLE;
    hadc1.Init.DiscontinuousConvMode = DISABLE;
    hadc1.Init.ExternalTrigConv = ADC_SOFTWARE_START;
    hadc1.Init.DataAlign = ADC_DATAALIGN_RIGHT;
    hadc1.Init.NbrOfConversion = 1;
    HAL_ADC_Init(&hadc1);

    /**Configure Regular Channel
        */
    sConfig.Channel = ADC_CHANNEL_TEMPSENSOR;
    sConfig.Rank = 1;
    sConfig.SamplingTime = ADC_SAMPLETIME_239CYCLES_5;
    HAL_ADC_ConfigChannel(&hadc1, &sConfig);
}

int main()
{
    MX_ADC1_Init();                                                 // initialize AD convertor
    while(HAL_ADCEx_Calibration_Start(&hadc1) != HAL_OK);           // calibrate AD convertor

    while(1) {        
        HAL_ADC_Start(&hadc1);                                      // start analog to digital conversion
        while(HAL_ADC_PollForConversion(&hadc1, 1000000) != HAL_OK);// wait for completing the conversion
        adcValue = HAL_ADC_GetValue(&hadc1);                        // read sensor's digital value
        vSense = adcValue * ADC_TO_VOLT;                            // convert sensor's digital value to voltage [V]
        /*
         * STM32F103xx Reference Manual:
         * 11.10 Temperature sensor
         * Reading the temperature, Page 235
         * Temperature (in °C) = {(V25 - Vsense) / Avg_Slope} + 25
         */
        temp = (V25 - vSense) / AVG_SLOPE + 25.0f;                  // convert sensor's output voltage to temperature [°C]
        pc.printf("temp = %3.1f%cC\n", temp, 176);                  // display chip's temperature [°C]
        led = !led;
        wait_ms(1000);
   }
}
