#ifndef HSCDTD008A_H
#define HSCDTD008A_H

#include "mbed.h"

#define RANGE   (2.4 + 2.4) // mT (+/- 2.4 mT)
#define RESOL   1024        // resolution = 1024 bit

/*
 * Status Register (STAT)
 */

// (Read Only)
#define DRDY    6           // Data Ready Detection
#define DOR     5           // Data Overrun Detection
#define FFU     2           // FIFO full alarm
#define TRDY    1           // Temperature ready
#define ORDY    0           // Offset ready

/*
 * FIFO Pointer Status Register (FFPT).
 */

// (Read Only)
#define FP  (0b00001111)    // Number of data in FIFO : 0 - 8

/*
 * Control 1 Register (CTRL1)
 */

// (Write/Read)
#define PC  7               // Power Mode Control 0 = Stand-by Mode (Default), 1 = Active Mode
#define ODR 3               // Output Data Rate Control in Normal State 00 = 0.5 Hz, 01 = 10Hz (Default), 10 = 20Hz, 11 = 100Hz
#define FS  1               // State Control in Active Mode 0 = Normal State,  1 = Force State (Default)

/*
 * Control 2 Register (CTRL2)
 */

// - When a CTRL2 register value was changed during the measurement,
//   The contents of the change are reflected after measurement.
// (Write/Read)
#define AVG 7               // Must be used default setting. 0 = (Default)
#define FCO 6               // Data storage method at FIFO. 0 = Direct (Default) , 1 = Comparison
#define AOR 5               // Choice of method of data Comparison at FIFO. 0 = OR(Default) , 1 = AND
#define FF  4               // FIFO Enable. 0 = Disable (Default) , 1 = Enable
#define DEN 3               // Data Ready Function Control Enable. 0 = Disabled (Default), 1 = Enabled
#define DRP 2               // DRDY signal active level control. 0 = ACTIVE LOW, 1 = ACTIVE HIGH (Default)
#define DTS 1               // Must be used default setting. 0 = (Default)
#define DOS 0               // Must be used default setting. 0 = (Default)

/*
 * Control 3 Register (CTRL3)
 */

// - Bit control at the same time is prohibited.
// - Priority of this register is MSB.
// (Write/Read)
#define SRST    7           // Soft Reset Control Enable. 0 = No Action (Default), 1 = Soft Reset, Note: return to zero after soft reset.
#define FRC     6           // Start to Measure in Force State. 0 = No Action (Default), 1 = Measurement Start, Note: return to zero after measurement.
#define STC     4           // Self Test Control Enable. 0 = No Action (Default), 1 = Set parameters to Self Test Response (STB) register., Note: return to zero immediately.
#define TCS     1           // Start to Measure Temperature in Active Mode. 0 = No Action (Default), 1 = Measurement Start
#define OCL     0           // Start to Calibrate Offset in Active Mode. 0 = No Action (Default), 1 = Action

/*
 * Control 4 Register (CTRL4)
 */

// - When a CTRL4 register value was changed during the measurement,
//   The contents of the change are reflected after measurement.
#define MMD     6           // Must be used default setting. 10 = (Default)
#define RS      4           // Set Dynamic range of output data. 0 = 14 bit signed value (-8192 to +8191) (Default), 1 = 15 bit signed value (-16384 to +16383)
#define AS      3           // Must be used default setting. 0 = (Default)
//
//
#define OK      0
#define ERROR   -1

// Function prototypes
void    printBinary(const unsigned char val);

class   HSCDTD008A
{
public:
    HSCDTD008A(PinName sda, PinName scl, uint8_t addr = 0x0C);
    int16_t toInt16(uint16_t word);
    void    standbyMode();
    void    normalMode(uint8_t odr = 0b01, bool enableDataReady = false);
    void    forcedMode();
    void    softReset();
    uint8_t selfTest();
    void    calibrateOffsets();
    void    setDriftOffsetX(uint16_t val);
    void    setDriftOffsetY(uint16_t val);
    void    setDriftOffsetZ(uint16_t val);
    void    compensateTemp();
    void    enableFifo();
    void    disableFifo();
    uint8_t getFifoPointer();
    bool    isFifoFull();
    bool    isFifoOverrun();
    bool    isDataReady();
    bool    getResolution();
    void    setResolution(bool fifteen_bits);
    uint8_t measure();
    void    readData();
    float   x();
    float   y();
    float   z();
private:
    I2C*            _i2c;
    const uint8_t   _addr;
    uint16_t        _x;
    uint16_t        _y;
    uint16_t        _z;
};
#endif // HSCDTD008A_H
