#include "Enigma.h"

/*$off*/
// Internal wiring of the five available rotors
const uint8_t Enigma::ROTOR_I[256] = {
    0xE0, 0xD7, 0xF1, 0x7A, 0x92, 0x84, 0x5E, 0x1B, 0xD6, 0x38, 0x79, 0x10, 0x7C, 0xA6, 0x1F, 0x4D,
    0x4F, 0x30, 0x19, 0xDD, 0x61, 0x00, 0xA2, 0x82, 0x98, 0xF9, 0x2F, 0x6A, 0xC5, 0x64, 0x9D, 0x9B,
    0xCA, 0x17, 0xD4, 0x24, 0x2B, 0x20, 0xD5, 0xEA, 0xBA, 0x25, 0x2E, 0x08, 0x78, 0x32, 0xAF, 0xE6,
    0x4C, 0x13, 0x56, 0xBD, 0xD9, 0xE5, 0x33, 0xEE, 0x69, 0xF3, 0x0F, 0x99, 0xBB, 0x4E, 0x01, 0x77,
    0x9C, 0x91, 0xF0, 0x89, 0xF8, 0x74, 0xB1, 0x95, 0x88, 0x42, 0xD0, 0x50, 0xFC, 0xA7, 0x7E, 0x7D,
    0x59, 0x73, 0x6D, 0x45, 0x5B, 0x3D, 0x3B, 0x34, 0x97, 0xED, 0xDE, 0x47, 0xF4, 0xE1, 0x60, 0xE4,
    0x2A, 0xAD, 0xFB, 0x81, 0xA9, 0x29, 0x57, 0x51, 0xC4, 0x3F, 0x53, 0x1D, 0xBE, 0x06, 0xFA, 0x85,
    0x12, 0x3A, 0x8A, 0x96, 0x63, 0x07, 0x8C, 0xD3, 0x72, 0xE3, 0x54, 0xCB, 0x4B, 0x35, 0x0B, 0xEB,
    0xDA, 0xD2, 0x3E, 0x7F, 0x1A, 0xB6, 0x0C, 0xC0, 0xB2, 0xC8, 0xA8, 0x66, 0x39, 0x8E, 0xF5, 0xA4,
    0xDF, 0xE2, 0x8D, 0x44, 0x31, 0x7B, 0x14, 0xDC, 0xA3, 0xC3, 0xFD, 0x1E, 0xEF, 0x23, 0x9A, 0x2D,
    0xF7, 0xB8, 0xC1, 0x94, 0x36, 0x5D, 0x48, 0x68, 0x80, 0xEC, 0x4A, 0x41, 0xB4, 0xE8, 0x16, 0x6E,
    0x49, 0x43, 0x18, 0x11, 0x9F, 0xD8, 0x8B, 0x83, 0xCC, 0x03, 0xCF, 0x5C, 0x27, 0xC6, 0x02, 0xE7,
    0xCD, 0x70, 0x5A, 0x04, 0xFF, 0xC7, 0xB0, 0x76, 0xB7, 0x26, 0xB3, 0xCE, 0x15, 0x6C, 0xB5, 0x0D,
    0x93, 0x09, 0xF2, 0xDB, 0xBC, 0x37, 0xAA, 0x87, 0xF6, 0xAB, 0x67, 0x21, 0x28, 0xC2, 0xC9, 0x52,
    0x0E, 0x5F, 0x58, 0xAE, 0x22, 0x6F, 0x86, 0x2C, 0x0A, 0x1C, 0x62, 0x9E, 0xA5, 0x46, 0x05, 0xAC,
    0x65, 0x40, 0x75, 0x6B, 0xB9, 0xE9, 0xD1, 0x8F, 0x71, 0xFE, 0x55, 0xA0, 0xA1, 0x3C, 0x90, 0xBF
};

const uint8_t Enigma::ROTOR_II[256] = {
    0xFF, 0xA2, 0x4F, 0x1A, 0x1B, 0x20, 0x9B, 0xEC, 0x47, 0x3E, 0xE7, 0x0D, 0x0F, 0x34, 0x50, 0x46,
    0x41, 0x9A, 0xF6, 0x63, 0xB6, 0x83, 0xEA, 0x51, 0x88, 0xD4, 0xC1, 0x52, 0x42, 0x37, 0xC0, 0xC6,
    0xE3, 0x1E, 0xF7, 0xA9, 0x60, 0x4B, 0xDA, 0x2D, 0x94, 0x7F, 0x85, 0x2F, 0xA0, 0x6A, 0xFA, 0xC8,
    0xDE, 0x06, 0x8D, 0x75, 0xF8, 0xEE, 0xAA, 0xC7, 0xEF, 0x38, 0x14, 0x69, 0x23, 0x05, 0x8C, 0x3D,
    0x5D, 0xC9, 0x13, 0x2E, 0xAB, 0x6C, 0x32, 0xC5, 0xFC, 0xB4, 0xB0, 0x98, 0x1D, 0xBC, 0x2A, 0xA6,
    0x3F, 0xF2, 0xAC, 0xC4, 0x8E, 0xBB, 0xF9, 0x49, 0xAD, 0x9C, 0xD5, 0x57, 0xEB, 0xA1, 0x3B, 0x15,
    0x7D, 0x28, 0x89, 0xB5, 0xB7, 0xE1, 0x5B, 0x65, 0xDC, 0x10, 0x82, 0x36, 0xB8, 0x2C, 0x25, 0x7A,
    0x91, 0x2B, 0xFE, 0x9F, 0x07, 0x01, 0xA4, 0x6F, 0xAE, 0x79, 0xF5, 0xD9, 0x48, 0x5C, 0x6D, 0x74,
    0xB1, 0x8A, 0x58, 0x33, 0xD0, 0x6E, 0x40, 0xBD, 0x4D, 0x97, 0xCF, 0x62, 0x84, 0x30, 0x67, 0xA7,
    0xE6, 0xE2, 0xB2, 0x61, 0x80, 0x92, 0x0E, 0x68, 0x4E, 0x66, 0x81, 0x16, 0x90, 0x53, 0x11, 0xBE,
    0x6B, 0xD1, 0x04, 0xD3, 0x08, 0x70, 0x95, 0x93, 0x4C, 0xA5, 0xDF, 0xD6, 0x5E, 0xA8, 0x7E, 0x9E,
    0x19, 0xE0, 0x77, 0x73, 0xBF, 0xF1, 0x21, 0x9D, 0x55, 0xFD, 0xE8, 0x56, 0xDB, 0xCE, 0x64, 0x43,
    0x0C, 0x29, 0x0B, 0x44, 0x31, 0xE9, 0x8F, 0x1F, 0x8B, 0x27, 0xED, 0x3A, 0xFB, 0x12, 0xCA, 0x59,
    0x1C, 0xDD, 0xB3, 0x87, 0x3C, 0x96, 0x78, 0x86, 0x54, 0xD2, 0xCC, 0x26, 0x99, 0xC3, 0x71, 0x02,
    0xF0, 0xE4, 0x45, 0x22, 0x18, 0xD8, 0x5F, 0xD7, 0xB9, 0xCD, 0xBA, 0x5A, 0x7B, 0xF4, 0xAF, 0xA3,
    0x00, 0x24, 0x35, 0x7C, 0x09, 0x39, 0x03, 0x17, 0xCB, 0xF3, 0x72, 0x4A, 0xE5, 0x76, 0xC2, 0x0A
};

const uint8_t Enigma::ROTOR_III[256] = {
    0x1C, 0xB8, 0x5A, 0xA6, 0xC2, 0xF9, 0x78, 0xF6, 0x41, 0x22, 0xD7, 0x1E, 0xAD, 0xCC, 0x96, 0xE0,
    0x59, 0x69, 0x09, 0x73, 0xE8, 0xFC, 0x1A, 0x17, 0xA2, 0x12, 0xFE, 0x00, 0xFD, 0x7E, 0x0C, 0x60,
    0x54, 0x8D, 0x2E, 0xFA, 0x9E, 0xB1, 0xA5, 0x3F, 0x03, 0x7B, 0xB9, 0xBF, 0x90, 0xDD, 0xE5, 0xD2,
    0x2F, 0x99, 0xCD, 0x80, 0x79, 0xCB, 0xC6, 0x5E, 0x31, 0x08, 0x29, 0xEA, 0xD9, 0x58, 0xF2, 0x4F,
    0x4E, 0x02, 0xC0, 0x98, 0x34, 0x47, 0x64, 0xB4, 0xCE, 0x5C, 0x8C, 0xA4, 0x28, 0x83, 0xCF, 0x44,
    0xEC, 0xC9, 0xF4, 0x30, 0x56, 0xB2, 0x1B, 0x8F, 0x07, 0xFF, 0x35, 0x6E, 0x0F, 0x7C, 0xAE, 0x11,
    0xAB, 0xE1, 0x13, 0xA0, 0xC8, 0xE9, 0x2C, 0x70, 0x82, 0x91, 0xEE, 0x76, 0xBB, 0xDE, 0x20, 0x7F,
    0xF0, 0x5B, 0xE4, 0x36, 0xA7, 0x2D, 0x89, 0x53, 0x93, 0x24, 0xA1, 0x3E, 0x51, 0xD0, 0xF8, 0x95,
    0xDB, 0x77, 0x8A, 0x40, 0x8B, 0xC3, 0xDC, 0x3D, 0xD4, 0x9F, 0x19, 0xBE, 0xD3, 0xE7, 0x45, 0x10,
    0x8E, 0x9D, 0x0D, 0x57, 0xA9, 0x72, 0x21, 0x49, 0x52, 0x04, 0xD8, 0xF1, 0xEB, 0x4C, 0xB0, 0x14,
    0x87, 0x55, 0x33, 0xA8, 0x61, 0x38, 0x39, 0x65, 0x42, 0x0E, 0xE2, 0xB5, 0x4B, 0x94, 0x27, 0xF7,
    0x06, 0x48, 0xD6, 0x6D, 0xB3, 0x50, 0x85, 0x6A, 0xBD, 0x3C, 0xDF, 0x6B, 0x67, 0x01, 0xFB, 0x86,
    0x63, 0xF3, 0xBC, 0x23, 0x88, 0xF5, 0xCA, 0x2B, 0x18, 0xC4, 0x3B, 0x5F, 0x84, 0x7D, 0xED, 0xDA,
    0x1D, 0xD1, 0x74, 0x0A, 0x43, 0x75, 0xC7, 0x32, 0x62, 0xB6, 0x16, 0xE6, 0x4A, 0xA3, 0x37, 0x71,
    0x81, 0x26, 0xAF, 0x68, 0x2A, 0x9A, 0x5D, 0x3A, 0x7A, 0xB7, 0xE3, 0x46, 0x0B, 0xD5, 0x25, 0x4D,
    0xC1, 0x66, 0xAA, 0x1F, 0x05, 0xBA, 0x9C, 0x6C, 0x9B, 0xAC, 0x97, 0x6F, 0xEF, 0xC5, 0x92, 0x15
};

const uint8_t Enigma::ROTOR_IV[256] = {
    0xC1, 0x1B, 0x04, 0xE6, 0x4A, 0x89, 0xAA, 0xBE, 0x78, 0x05, 0x8F, 0x1A, 0x02, 0x81, 0x45, 0x3A,
    0xAF, 0xEE, 0x68, 0xD4, 0x71, 0x5D, 0x92, 0x90, 0x36, 0x0D, 0x21, 0x14, 0x46, 0xBC, 0x09, 0xB2,
    0xEF, 0x8D, 0xBA, 0x8C, 0xAC, 0xDD, 0x01, 0xD8, 0xDA, 0xCF, 0xFF, 0xFC, 0xF6, 0x48, 0xF2, 0xB9,
    0x95, 0x59, 0xC4, 0x76, 0xB6, 0xCA, 0x58, 0x2C, 0x07, 0xEC, 0x2A, 0x47, 0x99, 0x8A, 0x31, 0x20,
    0xE2, 0x7F, 0x30, 0xBD, 0x8E, 0xB5, 0x70, 0x0B, 0x18, 0x37, 0x75, 0xB1, 0x94, 0xC7, 0x06, 0x00,
    0x73, 0xC8, 0xCC, 0x9E, 0xE9, 0x2D, 0x1D, 0x44, 0x98, 0x1F, 0x93, 0xFB, 0x41, 0x50, 0xAD, 0x32,
    0x61, 0xBB, 0xF1, 0xFE, 0x24, 0xCE, 0xB4, 0xAB, 0x72, 0xEA, 0x25, 0x67, 0xCD, 0x53, 0x26, 0x7A,
    0x77, 0xA8, 0x23, 0x19, 0x79, 0xD1, 0x9C, 0x5F, 0xF5, 0xC0, 0x1C, 0x5B, 0xE3, 0xC6, 0x6E, 0xD5,
    0xE0, 0x10, 0x3D, 0x66, 0x7D, 0x33, 0x51, 0x6C, 0x82, 0xD6, 0x84, 0x40, 0xDB, 0xE8, 0xC9, 0x62,
    0xF3, 0x5C, 0xD3, 0x60, 0x4D, 0x96, 0x08, 0x8B, 0xFA, 0xD0, 0xB7, 0x7E, 0xC2, 0x55, 0xB8, 0x63,
    0x86, 0x2F, 0x97, 0xA9, 0xA6, 0x3E, 0x34, 0xCB, 0x0E, 0x38, 0x6D, 0xBF, 0xF7, 0x11, 0x6B, 0x0A,
    0xEB, 0xF0, 0x15, 0x0C, 0xB3, 0x12, 0x88, 0xA4, 0x69, 0xD2, 0x9F, 0xA1, 0x9A, 0x39, 0x27, 0x2B,
    0x4F, 0x03, 0x91, 0xA5, 0x3B, 0xDE, 0xF9, 0x4B, 0x28, 0x9B, 0x1E, 0x17, 0xE7, 0x6F, 0xDF, 0x16,
    0x43, 0xF4, 0xDC, 0x7C, 0x87, 0x54, 0xE4, 0xA3, 0x52, 0x0F, 0xE5, 0x49, 0xED, 0x56, 0xFD, 0xE1,
    0x13, 0x22, 0x85, 0x9D, 0x4C, 0xD9, 0x65, 0x3C, 0x6A, 0x5E, 0xB0, 0xA0, 0x29, 0x2E, 0x42, 0x35,
    0xAE, 0x5A, 0x4E, 0xA7, 0xC3, 0x7B, 0xA2, 0x83, 0x64, 0x74, 0xF8, 0xD7, 0x80, 0x3F, 0xC5, 0x57
};

const uint8_t Enigma::ROTOR_V[256] = {
    0xFE, 0xC8, 0x49, 0x21, 0xA5, 0x5A, 0xF4, 0xA8, 0xC3, 0x1B, 0x9A, 0x5E, 0x22, 0xAE, 0x7F, 0xEA,
    0x8F, 0xF1, 0x15, 0x73, 0xE3, 0xCE, 0x05, 0x3C, 0x08, 0x07, 0xEB, 0x96, 0xCF, 0x98, 0x29, 0xBD,
    0x03, 0xAD, 0x9E, 0x32, 0x0E, 0xC1, 0xB9, 0xD2, 0x83, 0xD6, 0x4D, 0x14, 0x1C, 0x68, 0x9D, 0x31,
    0xA2, 0x97, 0x01, 0x7E, 0xE5, 0xF6, 0x2D, 0x27, 0xE8, 0xF5, 0xF9, 0xB2, 0x10, 0x82, 0xD8, 0x66,
    0x76, 0x8E, 0x19, 0xFD, 0xA3, 0xA6, 0x8A, 0xD0, 0x2F, 0xC9, 0xB4, 0x9C, 0xD7, 0x38, 0xDF, 0x99,
    0xEF, 0xC4, 0x09, 0xB1, 0x13, 0x3A, 0x6D, 0xBC, 0x58, 0x7C, 0x4C, 0x11, 0x7D, 0xE2, 0x30, 0xC7,
    0x2B, 0x6B, 0xAB, 0x6E, 0x63, 0x7B, 0xFF, 0x9B, 0x57, 0xD4, 0x61, 0x51, 0x26, 0x37, 0x4A, 0xED,
    0xB6, 0xAC, 0xC5, 0x92, 0x89, 0x91, 0xCC, 0x1F, 0x0C, 0xA0, 0x6F, 0x94, 0xEC, 0xA4, 0xDC, 0x4F,
    0x8B, 0xDB, 0x6C, 0x40, 0xC6, 0x48, 0x54, 0x45, 0x16, 0xE9, 0x6A, 0x20, 0x52, 0x67, 0x55, 0x34,
    0x90, 0x79, 0xF0, 0x33, 0x9F, 0xDA, 0x87, 0xF2, 0x85, 0xF3, 0x00, 0xF7, 0xBF, 0x69, 0xA9, 0x50,
    0x59, 0x36, 0xE0, 0x3D, 0x06, 0x0F, 0x46, 0x44, 0xB5, 0x56, 0xE7, 0x77, 0x3B, 0x5C, 0x0B, 0xE4,
    0x35, 0xC2, 0x72, 0x41, 0x2E, 0x0D, 0xAF, 0xF8, 0xC0, 0x0A, 0x17, 0x24, 0x95, 0x81, 0xEE, 0x1D,
    0x2A, 0xA7, 0xCB, 0xD1, 0x12, 0x02, 0xB7, 0x42, 0x84, 0x4E, 0x28, 0x78, 0x8C, 0x1A, 0xB0, 0x5B,
    0x7A, 0xFA, 0xDE, 0xBB, 0x93, 0x88, 0xCD, 0xE1, 0xFC, 0x04, 0x1E, 0x43, 0x8D, 0x74, 0x80, 0x25,
    0x47, 0xA1, 0xAA, 0x39, 0xDD, 0xB3, 0xD5, 0x3F, 0x65, 0x5D, 0xBA, 0x18, 0x71, 0x60, 0x75, 0x62,
    0x2C, 0x4B, 0x53, 0xCA, 0x64, 0xE6, 0x23, 0x5F, 0x3E, 0x86, 0x70, 0xFB, 0xD3, 0xD9, 0xBE, 0xB8
};

// Internal wiring of the reflector
const uint8_t Enigma::REFLECTOR[256] = {
    0x31, 0x74, 0x9C, 0xC4, 0x2A, 0x7E, 0xC6, 0x69, 0x19, 0xA3, 0x5D, 0x91, 0xE7, 0x15, 0x2F, 0xA1,
    0x29, 0x7C, 0xD1, 0x80, 0x9E, 0x0D, 0xF0, 0xDA, 0xFF, 0x08, 0xD5, 0xFC, 0xDF, 0xA0, 0x58, 0xAF,
    0x4D, 0xC3, 0x8F, 0x46, 0x97, 0x70, 0xCD, 0x48, 0xEA, 0x10, 0x04, 0xD3, 0x44, 0xD6, 0x51, 0x0E,
    0xFE, 0x00, 0xA2, 0x9D, 0x59, 0x50, 0x98, 0x6C, 0x53, 0x8C, 0xD8, 0x92, 0xA8, 0x87, 0x84, 0x6F,
    0x6D, 0xB6, 0x9A, 0xE3, 0x2C, 0x79, 0x23, 0xA7, 0x27, 0xA9, 0x66, 0xDB, 0xB5, 0x20, 0x68, 0x5B,
    0x35, 0x2E, 0xB1, 0x38, 0x89, 0xC7, 0xF1, 0x96, 0x1E, 0x34, 0xCB, 0x4F, 0x83, 0x0A, 0xF9, 0xFB,
    0x6B, 0xE2, 0xD0, 0x93, 0x71, 0x75, 0x4A, 0x8D, 0x4E, 0x07, 0x86, 0x60, 0x37, 0x40, 0xCC, 0x3F,
    0x25, 0x64, 0xE4, 0xC1, 0x01, 0x65, 0xFA, 0xB4, 0xF4, 0x45, 0xB2, 0xA5, 0x11, 0x7F, 0x05, 0x7D,
    0x13, 0xC8, 0xDD, 0x5C, 0x3E, 0x99, 0x6A, 0x3D, 0xF7, 0x54, 0xAD, 0xC9, 0x39, 0x67, 0xE8, 0x22,
    0xEF, 0x0B, 0x3B, 0x63, 0xF2, 0xD4, 0x57, 0x24, 0x36, 0x85, 0x42, 0xAB, 0x02, 0x33, 0x14, 0xED,
    0x1D, 0x0F, 0x32, 0x09, 0xD7, 0x7B, 0xB0, 0x47, 0x3C, 0x49, 0xFD, 0x9B, 0xE6, 0x8A, 0xE0, 0x1F,
    0xA6, 0x52, 0x7A, 0xBD, 0x77, 0x4C, 0x41, 0xBF, 0xF6, 0xBB, 0xD9, 0xB9, 0xDC, 0xB3, 0xC0, 0xB7,
    0xBE, 0x73, 0xD2, 0x21, 0x03, 0xCE, 0x06, 0x55, 0x81, 0x8B, 0xE5, 0x5A, 0x6E, 0x26, 0xC5, 0xEE,
    0x62, 0x12, 0xC2, 0x2B, 0x95, 0x1A, 0x2D, 0xA4, 0x3A, 0xBA, 0x17, 0x4B, 0xBC, 0x82, 0xEC, 0x1C,
    0xAE, 0xF5, 0x61, 0x43, 0x72, 0xCA, 0xAC, 0x0C, 0x8E, 0xF3, 0x28, 0xF8, 0xDE, 0x9F, 0xCF, 0x90,
    0x16, 0x56, 0x94, 0xE9, 0x78, 0xE1, 0xB8, 0x88, 0xEB, 0x5E, 0x76, 0x5F, 0x1B, 0xAA, 0x30, 0x18
};
/*$on*/

/**
 * @brief   Constructor
 * @note    Selects which rotors are going to be used and sets their intial position.
 * @note    Also connects pairs of plugs with cords on the plugboard.
 * @param
 * @retval
 */
Enigma::Enigma
(
    const uint8_t*  leftRotorSel,
    const uint8_t*  middleRotorSel,
    const uint8_t*  rightRotorSel,
    uint8_t         leftRotorPos,
    uint8_t         middleRotorPos,
    uint8_t         rightRotorPos,
    uint8_t         (*plugboard)[2],
    uint8_t         cords
) :
    _len(256),
    _leftRotorPos(leftRotorPos),
    _middleRotorPos(middleRotorPos),
    _rightRotorPos(rightRotorPos),
    _cords(cords)
{
    // Initialize entry wheel.
    for (size_t i = 0; i < _len; i++) {
        _entryWheel[i] = i;
    }

    // Select three rotors out of fives.
    _leftRotor = const_cast<uint8_t*>(leftRotorSel);
    _middleRotor = const_cast<uint8_t*>(middleRotorSel);
    _rightRotor = const_cast<uint8_t*>(rightRotorSel);

    // Initialize reflector.
    _reflector = const_cast<uint8_t*>(REFLECTOR);

    // Connect the plugboard
    _plugboard = new uint8_t[cords][2];
    for (size_t i = 0; i < cords; i++) {
        for (size_t j = 0; j < 2; j++) {
            _plugboard[i][j] = plugboard[i][j];
        }
    }
}

/**
 * @brief   Encrypts a byte array.
 * @note    It's capable to encrypt any byte, not only the 26 capital letters of the alphabet.
 * @retval  out Pointer to a byte array to store the encrypted bytes in.
 * @param   in  Pointer to a byte array to be encrypted.
 * @param   len Length of the byte array.
 * @retval  Pointer to a byte array containing the encrypted bytes.
 */
uint8_t* Enigma::encrypt(uint8_t* out, const uint8_t* in, size_t len)
{
    bool found;

    // Sets initial permutation
    int leftRotorPos = _leftRotorPos;
    int middleRotorPos = _middleRotorPos;
    int rightRotorPos = _rightRotorPos;

    for (size_t i = 0; i < len; i++) {
        uint8_t byte = in[i];

        // Swap bytes cross-connected with cords
        found = false;
        for (size_t j = 0; j < _cords; j++) {
            for (size_t k = 0; k < 2; k++) {
                if (_plugboard[j][k] == byte) {
                    byte = k == 0 ? _plugboard[j][1] : _plugboard[j][0];
                    found = true;
                    break;
                }
            }
            if (found) {
                break;
            }
        }

        // Advance right rotor on every byte entry
        rightRotorPos = _mod(rightRotorPos + 1);

        // Avance mid rotor using a notch located at position #13
        if ((rightRotorPos % 13) == 0) {
            middleRotorPos = _mod(middleRotorPos + 2);
        }

        // Advance left rotor using a notch located at position #201
        if ((middleRotorPos % 201) == 0) {
            leftRotorPos = _mod(leftRotorPos + 1);
        }

        // Pass through rotors
        byte = _rightRotor[_mod(byte + rightRotorPos)];
        byte = _middleRotor[_mod(byte + middleRotorPos - rightRotorPos)];
        byte = _leftRotor[_mod(byte + leftRotorPos - middleRotorPos)];

        // Pass through reflector
        byte = _reflector[_mod(byte - leftRotorPos)];
        byte = _entryWheel[_mod(byte + leftRotorPos)];

        // Inverse pass trough rotors
        byte = _entryWheel[_mod(_find(byte, _leftRotor) - leftRotorPos + middleRotorPos)];
        byte = _entryWheel[_mod(_find(byte, _middleRotor) - middleRotorPos + rightRotorPos)];
        byte = _entryWheel[_mod(_find(byte, _rightRotor) - rightRotorPos)];

        // Swap bytes cross-connected with cords
        found = false;
        for (size_t j = 0; j < _cords; j++) {
            for (size_t k = 0; k < 2; k++) {
                if (_plugboard[j][k] == byte) {
                    byte = k == 0 ? _plugboard[j][1] : _plugboard[j][0];
                    found = true;
                    break;
                }
            }
            if (found) {
                break;
            }
        }

        out[i] = byte;
    }

    return out;
}

/**
 * @brief   Decrypts a byte array.
 * @note    It's capable to encrypt any byte, not only the 26 capital letters of the alphabet.
 * @retval  out Pointer to a byte array to store the decrypted bytes in.
 * @param   in  Pointer to a byte array to be decrypted.
 * @param   len Length of the byte array.
 * @retval  Pointer to a byte array containing the decrypted bytes.
 */
uint8_t* Enigma::decrypt(uint8_t* out, const uint8_t* in, size_t len)
{
    return encrypt(out, in, len);
}

/**
 * @brief   Generates a new rotor wiring.
 * @note    Prints to the serial terminal. The generated byte array
 *          shall be then copy & pasted to his source code and recompiled.
 * @param   name Rotor's name. Should be "I", "II", "III", "IV" or "V"
 * @param   seed A number to seed the pseudo-ramdom generator with.
 * @retval
 */
void Enigma::genRotorWiring(const char* name, unsigned seed)
{
    uint8_t*    rotor = new uint8_t[_len];
    srand(seed);
    for (size_t i = 0; i < _len; i++) {
        rotor[i] = i;
    }
    for (int j = 0; j < (rand() % 1000 + 1000); j++) {
        for (size_t i = 0; i < _len; i++) {
            uint8_t byte = rotor[i];
            int     k = rand() % _len;
            rotor[i] = rotor[k];
            rotor[k] = byte;
        }
    }

    printf("const uint8_t Enigma::ROTOR_%s[256] = {\r\n", name);
    for (size_t i = 0; i < _len; i++) {
        if (i == 255) {
            printf("0x%.2X\r\n", rotor[i]);
        }
        else {
            switch (i % 16) {
                case 0:
                    printf("    0x%.2X, ", rotor[i]);
                    break;

                case 15:
                    printf("0x%.2X,\r\n", rotor[i]);
                    break;

                default:
                    printf("0x%.2X, ", rotor[i]);
            }
        }
    }

    printf("};\r\n\r\n");

    delete[] rotor;
}

/**
 * @brief   Generates a reflector wiring.
 * @note    Prints to the serial terminal. The generated byte array
 *          shall be then copy & pasted to his source code and recompiled.
 * @param   seed A number to seed the pseudo-ramdom generator with.
 * @retval
 */
void Enigma::genReflectorWiring(unsigned seed)
{
    uint8_t*    reflector = new uint8_t[_len];
    uint8_t*    modified = new uint8_t[_len];
    size_t      k;

    srand(seed);
    for (size_t i = 0; i < _len; i++) {
        reflector[i] = i;
    }

    memset(modified, 0, _len);

    for (size_t i = 0; i < _len; i++) {

        if (modified[i] != 0) {


            continue;
        }

        do {
            k = i + 1 + rand() % (_len - i - 1);
        } while (modified[k]);

        uint8_t byte = reflector[k];
        reflector[k] = reflector[i];
        reflector[i] = byte;
        modified[i] = 1;
        modified[k] = 1;
    }

    delete[] modified;

    printf("const uint8_t Enigma::REFLECTOR[256] = {\r\n");
    for (size_t i = 0; i < _len; i++) {
        if (i == 255) {
            printf("0x%.2X\r\n", reflector[i]);
        }
        else {
            switch (i % 16) {
                case 0:
                    printf("    0x%.2X, ", reflector[i]);
                    break;

                case 15:
                    printf("0x%.2X,\r\n", reflector[i]);
                    break;

                default:
                    printf("0x%.2X, ", reflector[i]);
            }
        }
    }

    printf("};\r\n\r\n");

    delete[] reflector;
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
size_t Enigma::_mod(long i)
{
    return((i % _len + _len) % _len);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
int Enigma::_find(uint8_t byte, uint8_t* array)
{
    for (size_t i = 0; i < _len; i++) {
        if (array[i] == byte) {
            return i;
        }
    }

    return -1;
}