/**
 * Demo for the Clock library <https://developer.mbed.org/users/hudakz/code/Clock/>
 *
 * Note: Software implemented Real Time Clock driven by a Ticker.
 *       No external hardware (like DS1307 or DS3231 or etc.) is needed.
 *
 */

#include "mbed.h"
#include "Clock.h"

Serial pc(USBTX, USBRX);


Clock rtc;    // Create an instance of Clock class (set to 00:00:00 January 1, 1970)

// Create alarms as needed
time_t  alarm1 = Clock::asTime(2015, 3, 24, 11, 36, 15);  // year, month (1 stands for Jan etc.), day of month, hour, minute, second
time_t  alarm2 = Clock::asTime(2015, 3, 24, 11, 37, 30);

volatile bool   ticked = false;  // tick flag

/**
 * @brief   Clock tick ISR - event handler (called once a second)
 * @note    Keep it as short as possible.
 * @param
 * @retval
 */
void onClockTick(void) {
    ticked = true;
}

/**
 * @brief   Main
 * @note
 * @param
 * @retval
 */
int main() {
    time_t      rawtime;
    struct tm*  p_tm;
    
    rtc.attach(onClockTick);  // attach a handler function to the rtc's tick event

    //
    // Set rtc for instance to 2015, March, 24th, 11 hours, 35 minutes, 45 seconds
    //
    rtc.set(2015, 3, 24, 11, 35, 45);
    //rawtime = time(NULL);        // if you like to call C library time function 
    rawtime = rtc.time();        // if you like to call Clock function
                    
    p_tm = gmtime(&rawtime);
    printf("Current time around the World:\r\n");
    printf("Phoenix, AZ (U.S.) : %d:%d\r\n", (p_tm->tm_hour -7) % 24, p_tm->tm_min);
    printf("Reykjavik (Iceland) : %d:%d\r\n", (p_tm->tm_hour + 0) % 24, p_tm->tm_min);
    printf("Beijing (China) : %d:%d\r\n", (p_tm->tm_hour + 8) % 24, p_tm->tm_min);



    while (true) {
        if(ticked == true) {
            ticked = false;  // clear the flag for next use in ISR

            // You draw clock hands on a graphical display
            // or update/refresh digital display or etc.
            // ...
            // In this demo we just display some info on the PC's serial terminal ..
            //
            pc.printf("==================================================\r\n");
            //rawtime = time(NULL);        // if you like you can call C library time function 
            rawtime = rtc.time();        // or Clock's function
            pc.printf("Time as seconds since January 1, 1970 = %d\r\n", rawtime);
            pc.printf("Time as a basic string = %s", ctime(&rawtime));
            
            //
            // Use custom format:
            //
            char buffer[32];
            strftime(buffer, 32, "%I:%M %p", localtime(&rawtime));
            pc.printf("Time as a custom formatted string = %s\r\n", buffer);

            //
            // Create your own format:
            //
            pc.printf("Date:  %.4d-%.2d-%.2d\r\n", rtc.year(), rtc.mon(), rtc.mday());
            pc.printf("Time:  %.2d:%.2d:%.2d\r\n", rtc.hour(), rtc.min(), rtc.sec());
            
            pc.printf("------------------------------------\r\n");

            //
            // Perform periodical tasks:
            //
            if(rtc.sec() % 10 == 0) {
                pc.printf("Called once a 10 seconds\r\n");
            }

            if(rtc.sec() == 0) {
                pc.printf("Called once a minute.\r\n");

                if(rtc.min() % 5 == 0)
                    pc.printf("Called once a 5 minutes\r\n");

                if(rtc.min() == 0) {
                    pc.printf("Called once an hour\r\n");

                    if(rtc.hour() % 3 == 0)
                        pc.printf("Called once a 3 hours\r\n");

                    if(rtc.hour() == 0) {
                        pc.printf("Called at midnight\r\n");

                        if(rtc.wday() == 3)
                            pc.printf("Called on Wednesday at midnight\r\n");
                    }
                }
            }

            //
            // Trigger the alarms:
            //
            if(rtc.time() == alarm1) {
                pc.printf("Alarm1 triggered!\r\n");
            }

            if(rtc.time() == alarm2) {
                pc.printf("Alarm2 triggered!\r\n");
            }
            
            pc.printf("------------------------------------\r\n");
        }
    }
}