/*
Using nRF24L01(+) as a Bluetooth Low Energy Advertiser/Broadcaster/Beacon
by hacking an nRF24L01(+) module (which is under $1 per unit on eBay).

See wiki page: <https://developer.mbed.org/users/hudakz/code/BLE_nRF24L01>

Note: It works with both nRF24L01 and nRF24L01+ modules.

Temperature readings measured by a DS1820 chip are broacasted over Bluetooth LE.
(You can easily modify the code to broadcast some other custom data.
 Only make sure not to exceed the 32 bytes packet length.)
 
The data can be received and displayed on an Android device (smartphone/tablet)
with Bluetooth Low Energy (Bluetooth Smart) enabled. 
In order to have Bluetooth 4.0 available (which implements Bluetooth LE)
your device should run Android 4.3 or more recent.
 */


#include "mbed.h"
#include "DS1820.h"

#define DEBUG 1

#if DEBUG
Serial serial(USBTX, USBRX);
#endif

// The MAC address of BLE advertizer -- just make one up
// If you decide to ceate more Beacons make sure that MAC address (MY_MAC) is unique for each beacon

#define MY_MAC_0    0x11
#define MY_MAC_1    0x12
#define MY_MAC_2    0x33
#define MY_MAC_3    0x44
#define MY_MAC_4    0x55
#define MY_MAC_5    0x66

#if defined(TARGET_LPC1768)
SPI         spi(p11, p12, p13);     // MOSI, MISO, SCK
DigitalOut  cs(p8);                 // CSN  (select SPI chip/slave)
DigitalOut  ce(p14);                // CE   (enable nRF24L01 chip)
DS1820      ds1820(p15);            // create a ds1820 sensor
#elif defined(TARGET_NUCLEO_F103RB) || defined(TARGET_NUCLEO_L152RE) || defined(TARGET_NUCLEO_F030R8)  \
   || defined(TARGET_NUCLEO_F401RE) || defined(TARGET_NUCLEO_F302R8) || defined(TARGET_NUCLEO_L053R8)  \
   || defined(TARGET_NUCLEO_F411RE) || defined(TARGET_NUCLEO_F334R8) || defined(TARGET_NUCLEO_F072RB)  \
   || defined(TARGET_NUCLEO_F091RC) || defined(TARGET_NUCLEO_F303RE) || defined(TARGET_NUCLEO_F070RB)  \
   || defined(TARGET_KL25Z ) || defined(TARGET_KL46Z) || defined(TARGET_K64F) || defined(TARGET_KL05Z) \
   || defined(TARGET_K20D50M) || defined(TARGET_K22F) \
   || defined(TARGET_NRF51822) \
   || defined(TARGET_RZ_A1H)
SPI         spi(D11, D12, D13);     // MOSI, MISO, SCK
DigitalOut  cs(D6);                 // CSN  (select SPI chip/slave)
DigitalOut  ce(D7);                 // CE   (enable nRF24L01 chip)
DS1820      ds1820(D8);             // create a ds1820 sensor

// If your board/plaform is not present yet then uncomment
// the following five lines and replace TARGET_YOUR_BOARD, SPI_MOSI, SPI_MISO, SPI_SCK, SPIS_CS, CE_PIN and DS1820_DATA_PIN as appropriate.

//#elif defined(TARGET_YOUR_BOARD)
//SPI         spi(SPI_MOSI, SPI_MISO, SPI_SCK);
//DigitalOut  cs(SPI_CS);             // CSN  (select SPI chip/slave)
//DigitalOut  cs(CE_PIN);             // CE   (enable nRF24L01+ chip)
//DS1820      ds1820(DS1820_DATA_PIN);// create a ds1820 sensor

#endif

uint8_t     buf[32];

/**
 * @brief   Implements CRC with LFSR
 * @note
 * @param   data:   packet data
 *          len:    packet length
 *          dst:    destination/location of CRC
 * @retval
 */
void bleCRC(const uint8_t* data, uint8_t len, uint8_t* dst) {
    uint8_t v, t, d;

    while(len--) {
        d = *data++;
        for(v = 0; v < 8; v++, d >>= 1) {
            t = dst[0] >> 7;
            dst[0] <<= 1;
            if(dst[1] & 0x80)
                dst[0] |= 1;
            dst[1] <<= 1;
            if(dst[2] & 0x80)
                dst[1] |= 1;
            dst[2] <<= 1;

            if(t != (d & 1)) {
                dst[2] ^= 0x5B;
                dst[1] ^= 0x06;
            }
        }
    }
}

/**
 * @brief   Reverses bit order in a single byte
 * @note
 * @param   a:  byte to be reveresed
 * @retval  byte with reveresed bit order
 */
uint8_t swapBits(uint8_t a) {
    uint8_t v = 0;
    if(a & 0x80)
        v |= 0x01;
    if(a & 0x40)
        v |= 0x02;
    if(a & 0x20)
        v |= 0x04;
    if(a & 0x10)
        v |= 0x08;
    if(a & 0x08)
        v |= 0x10;
    if(a & 0x04)
        v |= 0x20;
    if(a & 0x02)
        v |= 0x40;
    if(a & 0x01)
        v |= 0x80;
    return v;
}

/**
 * @brief   Implements whitening with LFSR
 * @note
 * @param   data:   location of the data to be whiten
 *          len:    data length
 *          whitenCoeff:    whitening coefficient
 * @retval
 */
void bleWhiten(uint8_t* data, uint8_t len, uint8_t whitenCoeff) {
    uint8_t m;
    while(len--) {
        for(m = 1; m; m <<= 1) {
            if(whitenCoeff & 0x80) {
                whitenCoeff ^= 0x11;
                (*data) ^= m;
            }

            whitenCoeff <<= 1;
        }

        data++;
    }
}

/**
 * @brief   Starts whitening
 * @note    the value we actually use is what BT'd use left shifted one...makes our life easier
 * @param   chan:   BT channel
 * @retval  single byte
 */
static inline uint8_t bleWhitenStart(uint8_t chan) {
    return swapBits(chan) | 2;
}

/**
 * @brief   Assembles the packet to be transmitted
 * @note
 * @param   data:   packet data
 *          len:    packet length
 *          dst:    BLE channel
 * @retval
 */
void blePacketEncode(uint8_t* packet, uint8_t len, uint8_t chan) {
    // Length is of packet, including crc. pre-populate crc in packet with initial crc value!
    uint8_t i, dataLen = len - 3;
    bleCRC(packet, dataLen, packet + dataLen);
    for(i = 0; i < 3; i++, dataLen++)
        packet[dataLen] = swapBits(packet[dataLen]);
    bleWhiten(packet, len, bleWhitenStart(chan));
    for(i = 0; i < len; i++)
        packet[i] = swapBits(packet[i]);    // the byte order of the packet should be reversed as well
}

/**
 * @brief   Sends cmommand to nRF24L01
 * @note
 * @param   cmd:    Command
 *          data:   Data associated with the command
 * @retval
 */
void nrfCmd(uint8_t cmd, uint8_t data) {

    // Write to nRF24's register

    cs = 0;
    spi.write(cmd);
    spi.write(data);
    cs = 1;
}

/**
 * @brief   Transfers one byte to nRF24L01
 * @note
 * @param   cmd: the byte to be transferred
 * @retval
 */
void nrfWriteByte(uint8_t cmd) {
    // transfer only one byte
    cs = 0;
    spi.write(cmd);
    cs = 1;
}

/**
 * @brief   Transfers several bytes to nRF24L01
 * @note
 * @param   data:   location of bytes to be transferred
 *          len:    number of bytes to be transferred
 * @retval
 */
void nrfWriteBytes(uint8_t* data, uint8_t len) {
    // transfer several bytes in a row
    cs = 0;
    do
    {
        spi.write(*data++);
    } while(--len);
    cs = 1;
}

int main() {
    static const uint8_t    chRf[] = { 2, 26, 80 };
    static const uint8_t    chLe[] = { 37, 38, 39 };
    
    uint8_t  i = 0;
    uint8_t  j = 0;
    uint8_t  ch = 0;
    uint8_t  data[4];  
    float*   temp = reinterpret_cast <float*>(&data[0]);
    int      error = 0;
    
    // Chip must be deselected
    cs = 1;
 
    // Setup the spi for 8 bit data, high steady state clock,
    // second edge capture, with a 10MHz clock rate
    spi.format(8,0);
    spi.frequency(10000000);
 
    ce = 0;

    // Initialize nRF24L01+, setting general parameters
    nrfCmd(0x20, 0x12);    // on, no crc, int on RX/TX done
    nrfCmd(0x21, 0x00);    // no auto-acknowledge
    nrfCmd(0x22, 0x00);    // no RX
    nrfCmd(0x23, 0x02);    // 4-byte address
    nrfCmd(0x24, 0x00);    // no auto-retransmit
    nrfCmd(0x26, 0x06);    // 1MBps at 0dBm
    nrfCmd(0x27, 0x3E);    // clear various flags
    nrfCmd(0x3C, 0x00);    // no dynamic payloads
    nrfCmd(0x3D, 0x00);    // no features
    nrfCmd(0x31, 32);      // always RX 32 bytes
    nrfCmd(0x22, 0x01);    // RX on pipe 0

    // Set access addresses (TX address in nRF24L01) to BLE advertising 0x8E89BED6
    // Remember that both bit and byte orders are reversed for BLE packet format
    buf[0] = 0x30;
    buf[1] = swapBits(0x8E);
    buf[2] = swapBits(0x89);
    buf[3] = swapBits(0xBE);
    buf[4] = swapBits(0xD6);
    nrfWriteBytes(buf, 5);
    buf[0] = 0x2A;          // set RX address in nRF24L01, doesn't matter because RX is ignored in this case
    nrfWriteBytes(buf, 5);
    
    if(!ds1820.begin()) {
#if DEBUG        
        serial.printf("No DS1820 sensor found!\r\n");
#endif
        return 1;
    }
    
          
    while(1) { 
        ds1820.startConversion();   // Start temperature conversion
        wait(1.0);                  // let DS1820 complete the temperature conversion
        error = ds1820.read(*temp); // read temperature from DS1820 and perform cyclic redundancy check (CRC)

#if DEBUG        
        switch(error) {
        case 0:    // no errors -> '*temp' contains the value of measured temperature
            serial.printf("temp = %3.1f\r\n", *temp);
            break;
        case 1:    // no sensor present -> '*temp' is not updated
            serial.printf("no sensor present\n\r");
            break;
        case 2:    // CRC error -> '*temp' is not updated
            serial.printf("CRC error\r\n");
        } 
#endif

        for(ch = 0; ch < (sizeof(chRf) / sizeof(*chRf)); ch++) {
            i = 0;
            buf[i++] = 0x42;            // PDU type, given address is random; 0x42 for Android and 0x40 for iPhone
            buf[i++] = 25;              // number of following data bytes, max 29  (CRC is not included)
            
            //----------------------------
            buf[i++] = MY_MAC_0;
            buf[i++] = MY_MAC_1;
            buf[i++] = MY_MAC_2;
            buf[i++] = MY_MAC_3;
            buf[i++] = MY_MAC_4;
            buf[i++] = MY_MAC_5;
        
            buf[i++] = 2;               // flags (LE-only, limited discovery mode)
            buf[i++] = 0x01;
            buf[i++] = 0x05;
        
            buf[i++] = 9;               // length of the name, including type byte
            buf[i++] = 0x08;            // TYPE_NAME_SHORT
            buf[i++] = 'n';
            buf[i++] = 'R';
            buf[i++] = 'F';
            buf[i++] = '2';
            buf[i++] = '4';
            buf[i++] = 'L';
            buf[i++] = '0';
            buf[i++] = '1';
        
            buf[i++] = 5;               // length of custom data, including type byte
            buf[i++] = 0xff;            // TYPE_CUSTOMDATA
    
            buf[i++] = data[0];         // temperature floating point value (four bytes)
            buf[i++] = data[1];         
            buf[i++] = data[2];         
            buf[i++] = data[3];         
            //----------------------------
            
            buf[i++] = 0x55;            // CRC start value: 0x555555
            buf[i++] = 0x55;
            buf[i++] = 0x55;
        
            nrfCmd(0x25, chRf[ch]);
            nrfCmd(0x27, 0x6E);         // Clear flags
            blePacketEncode(buf, i, chLe[ch]);
            nrfWriteByte(0xE2);         // Clear RX Fifo
            nrfWriteByte(0xE1);         // Clear TX Fifo
            
            cs = 0;
            spi.write(0xA0);
            for(j = 0; j < i; j++)
                spi.write(buf[j]);
            cs = 1;
        
            nrfCmd(0x20, 0x12);         // TX on
            ce = 1;                     // Enable Chip
            wait_ms(50);    
            ce = 0;                     // (in preparation of switching to RX quickly)
        }
    }
}
