/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "ble/BLE.h"
#include "LEDService.h"


DigitalOut alivenessLED(LED1, 0);
DigitalOut actuatedLED(LED2, 0);
Serial pc(USBTX, USBRX);

const static char     DEVICE_NAME[] = "LEDWTFFTW";
static const uint16_t uuid16_list[] = {LEDService::LED_SERVICE_UUID};
const static char AdvData[]="Hello World!";
LEDService *ledServicePtr;

Ticker ticker;

void timeoutCallback(const Gap::TimeoutSource_t source){
    BLE::Instance().gap().stopAdvertising();
    pc.printf("got a timeout\r\n");
    BLE::Instance().gap().startAdvertising();
}
    

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    pc.printf("disconnected\r\n");
    BLE::Instance().gap().startAdvertising();
    uint8_t deviceName[30] = {0};
    unsigned lengthP = 0;
    BLE::Instance().gap().getDeviceName(deviceName, &lengthP);
    pc.printf("new device name is %s", deviceName);

}

void connectionCallback(const Gap::ConnectionCallbackParams_t *params){
    
    pc.printf("connected\r\n");
    
}
void periodicCallback(void)
{
//    pc.printf("got here\r\n");
    alivenessLED = !alivenessLED; /* Do blinky on LED1 to indicate system aliveness. */
}

/**
 * This callback allows the LEDService to receive updates to the ledState Characteristic.
 *
 * @param[in] params
 *     Information about the characterisitc being updated.
 */
void onDataWrittenCallback(const GattWriteCallbackParams *params) {
    if ((params->handle == ledServicePtr->getValueHandle()) && (params->len == 1)) {
        actuatedLED = *(params->data);
    }
}

 
/**
 * This function is called when the ble initialization process has failed
 */
void onBleInitError(BLE &ble, ble_error_t error)
{
    /* Initialization error handling should go here */
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }
 
    ble.gap().onDisconnection(disconnectionCallback);
    ble.gattServer().onDataWritten(onDataWrittenCallback);
    ble.gap().onTimeout(timeoutCallback);

    bool initialValueForLEDCharacteristic = false;
    ledServicePtr = new LEDService(ble, initialValueForLEDCharacteristic);

    /* setup advertising */
    ble.gap().setDeviceName((uint8_t*)DEVICE_NAME);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA, (uint8_t *)AdvData, sizeof(AdvData));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(60000); /* you shall not pass */
    
    ble.gap().setAdvertisingTimeout(900);
    ble.gap().startAdvertising();
}

int main(void)
{
    ticker.attach(periodicCallback, 1); /* Blink LED every second */

    BLE &ble = BLE::Instance();
    ble.init(bleInitComplete);

    pc .baud(19200);
    /* SpinWait for initialization to complete. This is necessary because the
     * BLE object is used in the main loop below. */
    while (ble.hasInitialized()  == false) { /* spin loop */ }

    while (true) {
//        pc.printf("got here\r\n");
        ble.waitForEvent();
    }
}
