/*
 * Licence: Apache2 Licence
 * Author: Yuichi Tateno 
 *
 * base code from https://lowreal.net/2016/03/18/1 <Author: cho45>
 */

#ifndef _MAX_7129_HPP
#define _MAX_7129_HPP

#include <mbed.h>

class MAX7219 {
  SPI spi;
    DigitalOut cs;

public:
    typedef enum {
        NOOP = 0b0000,
        DIGIT0 = 1,
        DIGIT1 = 2,
        DIGIT2 = 3,
        DIGIT3 = 4,
        DIGIT4 = 5,
        DIGIT5 = 6,
        DIGIT6 = 7,
        DIGIT7 = 8,
        DECODE_MODE = 0b1001,
        INTENSITY = 0b1010,
        SCAN_LIMIT = 0b1011,
        SHUTDOWN = 0b1100,
        DISPLAY_TEST = 0b1111,
    } ADDRESS;

    typedef enum {
        SHUTDOWN_MODE_SHUTDOWN = 0b0,
        SHUTDOWN_MODE_NORMAL = 0b1,
    } SHUTDOWN_MODE;

    typedef enum {
        NO_DECODE_FOR_DIGITS_7_0 = 0b00000000,
        CODE_B_DECODE_FOR_DIGIT_0_NO_DECODE_FOR_DIGITS_7_1 = 0b00000001,
        CODE_B_DECODE_FOR_DIGIT_3_0_NO_DECODE_FOR_DIGITS_7_4 = 0b00001111,
        CODE_B_DECODE_FOR_DIGIT_7_0 = 0b11111111,
    } DECODE_MODE_TYPE;

    MAX7219(SPI _spi, PinName _cs_pin) :
        spi(_spi), cs(_cs_pin, 0)
    {
    }

    void write(uint8_t address, uint8_t data) {
    cs = 0;
    spi.write(address);
    spi.write(data);
    cs = 1;
    }

    void begin() {
        setDecodeMode(CODE_B_DECODE_FOR_DIGIT_7_0);
        setScanLimit(7);
        setShutdown(false);

        setIntensity(8);

        for (uint8_t i = 0; i < 8; i++) {
            write(i + 1, 1);
        }
    }

    void setDecodeMode(DECODE_MODE_TYPE mode) {
        write(DECODE_MODE, mode);
    }

    void setScanLimit(uint8_t limit) {
        write(SCAN_LIMIT, limit);
    }

    void setIntensity(uint8_t intensity) {
        write(INTENSITY, intensity);
    }

    void setShutdown(bool shutdown) {
        write(SHUTDOWN, shutdown ? SHUTDOWN_MODE_SHUTDOWN : SHUTDOWN_MODE_NORMAL);
    }

    void print(const char* buf) {
        size_t len = strlen(buf);
        uint8_t digit = 0, dp = 0;
        for (size_t i = 0; i < len && digit < 8; i++) {
            const char c = buf[len - i - 1];
            if (c == '.') {
                dp = 1<<7;
                continue;
            } 

            uint8_t b;
            if (c == '-') {
                b = 0b1010;
            } else
            if (c == ' ') {
                b = 0b1111;
            } else
            if ('0' <= c && c <= '9') {
                b = static_cast<uint8_t>(c - '0');
            } else {
                continue;
            }

            write(digit + 1, b | dp);
            dp = 0;
            digit++;
        }

        for (; digit < 8; digit++) {
            write(digit + 1, 0b1111);
        }
    }
};


#endif // _MAX_7129_HPP
