/*
* Software License Agreement (BSD License)
*
* Copyright (c) 2015, University of York Robotics Laboratory (YRL).
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
*
* * Redistributions of source code must retain the above copyright
* notice, this list of conditions and the following disclaimer.
* * Redistributions in binary form must reproduce the above
* copyright notice, this list of conditions and the following
* disclaimer in the documentation and/or other materials provided
* with the distribution.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
* FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
* COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
* LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
* ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
*/
/**
* @file PiSwarmControllerFunctions.cpp
* @brief Definitions of functions used in the Pi-Swarm Controller
* @details This file contains functions used in the Pi-Swarm controller
* @version 1.0
* @author Robert
* @author Evans
* @date 24/07/15
*/
#include "pi_swarm_functions.h"

//Piswarm variables
extern PiSwarm piswarm;
extern float BASE_SPEED;
extern int16_t g_currentHeading;

//Timers
extern Timer turn_timer;
extern Timer levy_timer;
extern Timer back_move_timer;

//State
extern uint8_t volatile gv_state;
extern uint8_t consecutive_turn_count;
extern uint8_t volatile sync_attempt_count;

//Beacon on/off ticks
extern int8_t g_beaconOn;
extern int8_t volatile tick_beacon_suspected;
extern int8_t tick_beacon_period_check;
extern uint8_t volatile beacon_detected[8];
extern uint8_t beacon_syncronised_flag;

//Levy walk variables
extern uint32_t levy_target_time_us;
extern uint8_t set_new_levy_time_flag;

//This function should be used to change between states in the finite state machine
//the function is used to ensure ticks,flags and timer variables are reset where necessary. 
void changeState(uint8_t newState){
    switch(newState){
        case States::READY_TO_START:
            gv_state = States::READY_TO_START;
            break;
        
        case States::SEARCHING_FWD: //A new Levy Path Target time is set before the state is changed
            levy_timer.start();
            //configure leds
            piswarm.set_oled_colour(0,255,0);
            piswarm.set_oleds(1,1,1,1,1,1,1,1,1,1);
            
            //Reset the beacon ticks unless changing from state 12
            g_beaconOn = 100;
            if(gv_state != States::SEARCHING_TURNING){
                tick_beacon_suspected = 100;
                tick_beacon_period_check = 100;
            }
            
            turn_timer.reset();
            if(set_new_levy_time_flag == 1){
                uint16_t randNum = rand() % 1000; //generate random number            
                float levyDistance = randToLevy(randNum); //convery random number into levy path length using levy distribution
                levy_target_time_us = levyDistToTime(levyDistance); //set the levyTargetTime variable      
                levy_timer.reset();
                set_new_levy_time_flag = 0;
                
            }
            gv_state = States::SEARCHING_FWD;
            break; 
          
        case States::SEARCHING_TURNING:
            //In case piswarm is stuck somewhere then do backward manoevre
            static int8_t fwd_or_bck = 1;
            
            if(consecutive_turn_count > 5){
                piswarm.left_motor(0.2*fwd_or_bck);
                piswarm.right_motor(0.2*fwd_or_bck);
                wait(1);
                consecutive_turn_count = 0;
                fwd_or_bck = fwd_or_bck * -1;
            } else {
                consecutive_turn_count++;
            }
            levy_timer.stop();
            gv_state = States::SEARCHING_TURNING;
            break;
            
        
        case States::MOVING_TO_BEACON:
            back_move_timer.start();
            back_move_timer.reset();
            sync_attempt_count = 0;
            piswarm.stop();
            piswarm.cls(); 
            piswarm.printf("S2");
            //set the piswarm leds to show it is in state 2
            piswarm.set_oled_colour(0,0,255);
            beacon_syncronised_flag = 0;
            gv_state = States::MOVING_TO_BEACON;
            break;   
        case States::AT_TARGET_BEACON:
            
            piswarm.stop();
            piswarm.cls(); 
            piswarm.printf("S3");
            piswarm.set_oled_colour(255,0,255); 
            gv_state = States::AT_TARGET_BEACON;
            break;
        case States::AT_HOME_BEACON:
            
            piswarm.stop();
            piswarm.cls(); 
            piswarm.printf("S4");
            gv_state = States::AT_HOME_BEACON;
            break;            
    }
}

//Converts the random num to a distance in the chosen Levy Distribution
float randToLevy(uint16_t randomNumber){
    int16_t rangeMarkers[23] = {149,266,361,440,507,565,615,659,698,733,765,794,820,844,866,887,906,924,941,957,972,986,999};
    int8_t loopCounter = 0;
    
    //Currently set up so shortest path length is 1.5m and the longest is 7m.
    float pathLength = 1.5;
    
    for(loopCounter = 0; loopCounter <= 22; loopCounter++){
        if(randomNumber <= rangeMarkers[loopCounter]){
            return pathLength;
        } else {
            pathLength = pathLength + 0.25;
        }
    } 
    return -1;     
}

//This converts the levy Distance into a time in us it will take the robot tomove this distance.
uint32_t levyDistToTime(float distance){
    //Time = Distance / Speed
    uint32_t time = distance * 1000000;
    //Convert to us 
    time = time / BASE_SPEED;
    return time;
} 
    
    
void turnDegrees(int16_t degrees){
    piswarm.stop();
    wait_ms(100);
    uint16_t endCount = 0;
    
    //calculated 
    endCount =  3.8 * abs(degrees);
    if(degrees > 180){
        degrees  = degrees - 360;
    } else if (degrees < -180){
        degrees = degrees + 360;
    }
    turn_timer.reset();
    while(turn_timer.read_ms() <= endCount){        
        if (degrees >= 0 ){
            piswarm.left_motor(-0.2);
            piswarm.right_motor(0.2);
        } else {
            piswarm.left_motor(0.2);
            piswarm.right_motor(-0.2);
        }
    }
    g_currentHeading = g_currentHeading + degrees; //Update the current heading
    
    if(g_currentHeading > 180){
        g_currentHeading = g_currentHeading - 360;
    } else if (g_currentHeading < -180){
         g_currentHeading = g_currentHeading + 360;
    }
    piswarm.stop();
}

void calculateNewHeading(){
    
    char byte = (beacon_detected[0] << 7) + (beacon_detected[1] << 6) + (beacon_detected[2] << 5) + (beacon_detected[3] << 4) +
    (beacon_detected[4] << 3) + (beacon_detected[5] << 2) + (beacon_detected[6] << 1) + (beacon_detected[7]);
    
    switch(byte){
        
        //single sensor cases---------------------------
        case 0x80://10000000
            piswarm.cls();     
            g_currentHeading = -15; 
            piswarm.printf("%d",g_currentHeading);
            break; 
            
        case 0x40://01000000
            piswarm.cls();     
            g_currentHeading = -45; 
            piswarm.printf("%d",g_currentHeading);
            break; 
            
        case 0x20://00100000
            piswarm.cls();     
            g_currentHeading = -90; 
            piswarm.printf("%d",g_currentHeading);
            break; 
            
         case 0x10://00010000
            piswarm.cls();     
            g_currentHeading = -144; 
            piswarm.printf("%d",g_currentHeading);
            break;
            
         case 0x08://00001000
            piswarm.cls();     
            g_currentHeading = 144; 
            piswarm.printf("%d",g_currentHeading);
            break;
            
        case 0x04://00000100
            piswarm.cls();     
            g_currentHeading = 90; 
            piswarm.printf("%d",g_currentHeading);
            break;
            
        case 0x02://00000010
            piswarm.cls();     
            g_currentHeading = 45; 
            piswarm.printf("%d",g_currentHeading);
            break;
            
        case 0x01://00000001
            piswarm.cls();     
            g_currentHeading = 15; 
            piswarm.printf("%d",g_currentHeading);
            break;
            
        //Two sensors detect beacon---------------------------
        case 0xC0://11000000
            piswarm.cls();     
            g_currentHeading = -30; 
            piswarm.printf("%d",g_currentHeading);
            break;
            
        case 0x60://01100000
            piswarm.cls();     
            g_currentHeading = -68; 
            piswarm.printf("%d",g_currentHeading);
            break;
            
        case 0x06://00000110
            piswarm.cls();     
            g_currentHeading = 68; 
            piswarm.printf("%d",g_currentHeading);
            break;
        
        case 0x03://00000011
            piswarm.cls();     
            g_currentHeading = 30; 
            piswarm.printf("%d",g_currentHeading);
            break;
            
        case 0x81://10000001
            piswarm.cls();     
            g_currentHeading = 0; 
            piswarm.printf("%d",g_currentHeading);
            break;

        //3 sensor special cases
         case 0x83://10000011
            piswarm.cls();     
            g_currentHeading = 20; 
            piswarm.printf("%d",g_currentHeading);
            break;
            
        case 0xC1://11000001
            piswarm.cls();     
            g_currentHeading = -20; 
            piswarm.printf("%d",g_currentHeading);
            break;
    
        //Calculated cases ------------------------------
        case 0xC3://11000011
            piswarm.cls();     
            g_currentHeading = 0; 
            piswarm.printf("%d",g_currentHeading);
            break; 
            
        case 0xE3://11100011
            g_currentHeading = -23;
            piswarm.cls(); 
            piswarm.printf("%d",g_currentHeading);
            break; 
            
        case 0xE1://11100001
            g_currentHeading = -43;
            piswarm.cls(); 
            piswarm.printf("%d",g_currentHeading); 
            break; 
            
        case 0xE0://11100000
            g_currentHeading = -65;
            piswarm.cls(); 
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0xF0://11110000
            g_currentHeading = -80;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x78://01111000
            g_currentHeading = -140;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
         case 0x1E://00011110
            g_currentHeading = 140;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
            
        case 0x3C://00111100
            g_currentHeading = -180;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0xF8://11111000
            g_currentHeading = -110;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x7C://01111100
            g_currentHeading = -130;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x3E://00111110
            g_currentHeading = 130;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
             
        case 0x1F://00011111
            g_currentHeading = 110;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0xF1://11110001
            g_currentHeading = -75;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x8F://10001111
            g_currentHeading = 60;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0xC7://11000111
            g_currentHeading = 23;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x70://01110000
            g_currentHeading = -103;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x30://00110000
            g_currentHeading = -131;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x38://00111000
            g_currentHeading = -153;
            piswarm.cls(); 
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x18://00011000
            g_currentHeading = 180;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x1C://00011100
            g_currentHeading = 153;
            piswarm.cls(); 
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x0C://00001100
            g_currentHeading = 131;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x0E://00001110
            g_currentHeading = 103;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x0F://00001111
            g_currentHeading = 80;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x07://00000111
            g_currentHeading = 65;
            piswarm.cls(); 
            piswarm.printf("%d",g_currentHeading);
            break;
            
        case 0x87://10000111
            g_currentHeading = 43;
            piswarm.cls(); 
            piswarm.printf("%d",g_currentHeading); 
            break;
            
        case 0x00://00000000
            //g_currentHeading = 30;
            piswarm.cls();  
            piswarm.printf("%d",g_currentHeading); 
            break;
        default :
            piswarm.cls(); 
            piswarm.printf("Def");
    }          
}

int16_t mod16 (int16_t a,int16_t b){
    int16_t ret = a % b;
    if(ret < 0){
        ret = ret + b;
    }
    return ret;
}
    
int8_t mod8 (int8_t a,int8_t b){
    int8_t ret = a % b;
    if(ret < 0){
        ret = ret + b;
    }
    return ret;
}
    