
#include "mbed.h"
#include "math.h"

#ifndef __MBED_I2SSLAVE_H__
#define __MBED_I2SSLAVE_H__

extern FunctionPointer akjnh3489v8ncv;


class I2SSlave{
	public:
		/* @param tx_sda Transmitter serial data line
		 * @param tx_ws Transmitter word select line
		 * @param clk Shared transmitter/receiver clock line
		 * @param rx_sda Receiver serial data line
		 * @param rx_ws Receiver word select line */
		I2SSlave(PinName tx_sda, PinName tx_ws, PinName clk, PinName rx_sda, PinName rx_ws);

		// bit Set the number of bits per write
		// STEREO (0) or MONO (1) mode
		void format(int bit, bool mode);
		
		void Write(int32_t aLeft, int32_t aRight);

		void Read(int32_t* aLeft, int32_t* aRight);
		
		// Mode to enable - NONE, TRANSMIT only, RECEIVE only, BOTH
		// Enables tx/rx interrupts
		void start(int mode);
		
		void Start();
		void Stop();
		
		void attach(void(*fptr)(void)){
				akjnh3489v8ncv.attach(fptr);
		}
		
		template<typename T>
		void attach(T *tptr, void(T::*mptr)(void)){
				akjnh3489v8ncv.attach(tptr, mptr);
		}
        
		// bit0: receive/transmit interrupt active
		// bit1: receive/transmit DMA request 1
		// bit2: receive/transmit DMA request 2
		// bit[11:8]: receive FIFO level
		// bit[19:16]: transmit FIFO level
		int status();

		int GetRxLevel();
		int GetTxLevel();
		
		//defines
    #define STEREO  0
    #define MONO    1
    
    #define I2SFIFO_EMPTY    0
    #define I2SFIFO_FULL     8
    
    #define RAM_LENGTH  1024
    #define RAM_LIMIT   (RAM_LENGTH - 1) 
    #define PTR_MAX     ((RAM_LENGTH / 8) - 1)
    
    #define NONE        0
    #define TRANSMIT    1
    #define RECEIVE     2
    #define BOTH        3

	private:        
        /** I2S intitalize function
         *
         * @param tx_sda Transmitter serial data line
         * @param tx_ws Transmitter word select line
         * @param clk Shared transmitter/receiver clock line
         * @param rx_sda Receiver serial data line
         * @param rx_ws Receiver word select line
         * @return Returns 0 for successful initialisation, -1 for an error
         */
        int initialize_(PinName tx_sda, PinName tx_ws, PinName clk, PinName rx_sda, PinName rx_ws);
        /** Set internal clock divide by rate
         *
         * @param divideBy Divide by 1, 2, 4 or 8
         */
        void setClocks_(int divideBy);
        /** Set up the pins on the processor itself
         *
         * @param tx_sda Transmitter serial data line
         * @param tx_ws Transmitter word select line
         * @param clk Shared transmitter/receiver clock line
         * @param rx_sda Receiver serial data line
         * @param rx_ws Receiver word select line
         */
        void setPins_(PinName tx_sda, PinName tx_ws, PinName clk, PinName rx_sda, PinName rx_ws);
        /** Set the data transmission format 
         *
         * @param bit Set the number of bits per write
         * @param mode Set STEREO (0) or MONO (1) mode
         */
        void format_(int bit, bool mode);
        
				// Set slave mode
        void modeConfig_(void);
        
				/** Store PinName values
         * 
         * @param tx_sda Transmitter serial data line
         * @param tx_ws Transmitter word select line
         * @param clk Shared transmitter/receiver clock line
         * @param rx_sda Receiver serial data line
         * @param rx_ws Receiver word select line
         */
        void storePins_(PinName tx_sda, PinName tx_ws, PinName clk, PinName rx_sda, PinName rx_ws);
        //variables
        int bit_;
        bool mode_;
        PinName tx_sda_; 
        PinName tx_ws_; 
        PinName clk_;
        PinName rx_sda_;
        PinName rx_ws_;
};

#endif /*__MBED_I2S_H__*/


