
#ifndef MBED_TLV320_H
#define MBED_TLV320_H

#include "mbed.h"
#include "I2SSlaveHL.h"

class TLV320 {
    public:
        // sda Serial data pin (p9 or p28)
        // scl Serial clock pin (p10 or p27)
        // addr Object address
        TLV320(PinName sda, PinName scl, int addr, I2SSlave* aI2S);
				
				void Init(int aFrequ);
        
        // 0 = power down, 1 = power up
				void power(bool powerUp);
        
				/** Overloaded power() function default = 0x07, record requires 0x02
				* @param device Call individual devices to power up/down
        * Device power      0x00 = On 0x80 = Off
        * Clock             0x00 = On 0x40 = Off
        * Oscillator        0x00 = On 0x20 = Off
        * Outputs           0x00 = On 0x10 = Off
        * DAC               0x00 = On 0x08 = Off
        * ADC               0x00 = On 0x04 = Off
        * Microphone input  0x00 = On 0x02 = Off
        * Line input        0x00 = On 0x01 = Off */
        void power(int device);
        
				/** Set I2S interface bit length and mode
        * @param length Set bit length to 16, 20, 24 or 32 bits
        * @param mode Set STEREO (0), MONO (1) */
        void format(char length, bool mode);
				
				// Returns an integer 0 = success, -1 = unrecognnised frequency
				// The TLV320 supports the following frequencies: 8kHz, 8.021kHz, 32kHz, 44.1kHz, 48kHz, 88.2kHz, 96kHz
        // Default is 44.1kHz
				int frequency(int hz);
        
				// Reset TLV320
				void reset(void);
				
				/** Line in volume control i.e. record volume
        * @return Returns 0 for success, -1 if parameters are out of range
        * Parameters accept a value, where 0.0 < parameter < 1.0 and where 0.0 maps to -34.5dB 
        * and 1.0 maps to +12dB (0.74 = 0 dB default). */
        int inputVolume(float leftVolumeIn, float rightVolumeIn);
        
				/** Headphone out volume control
        * @return Returns 0 for success, -1 if parameters are out of range
        * Parameters accept a value, where 0.0 < parameter < 1.0 and where 0.0 maps to -73dB (mute) 
        * and 1.0 maps to +6dB (0.5 = default) */
        int outputVolume(float leftVolumeOut, float rightVolumeOut);
				
				// Analog audio path control
        // @param bypassVar Route analogue audio direct from line in to headphone out
        void bypass(bool bypassVar);
        
				// Digital audio path control
        // @param softMute Mute output
        void mute(bool softMute);

		protected:
        char cmd[2]; //the address and command for TLV320 internal registers
        int mAddr;   //register write address
		private:
        I2C mI2c_;      
        I2SSlave* mI2S;
		
				/** Sample rate control
        * @param rate Set the sampling rate as per datasheet section 3.3.2
        * @param clockIn Set the clock in divider MCLK, MCLK_DIV2
        * @param clockMode Set clock mode CLOCK_NORMAL, CLOCK_USB */
        void setSampleRate_(char rate, bool clockIn, bool mode, bool bOSR); 
				
				// Digital interface activation
        void activateDigitalInterface_(void);
        
				// Digital interface deactivation
				void deactivateDigitalInterface_(void);

        //TLV320AIC23B register addresses as defined in the TLV320AIC23B datasheet
        #define LEFT_LINE_INPUT_CHANNEL_VOLUME_CONTROL  (0x00 << 1)
        #define RIGHT_LINE_INPUT_CHANNEL_VOLUME_CONTROL (0x01 << 1)
        #define LEFT_CHANNEL_HEADPHONE_VOLUME_CONTROL   (0x02 << 1)
        #define RIGHT_CHANNEL_HEADPHONE_VOLUME_CONTROL  (0x03 << 1)
        #define ANALOG_AUDIO_PATH_CONTROL               (0x04 << 1)
        #define DIGITAL_AUDIO_PATH_CONTROL              (0x05 << 1)
        #define POWER_DOWN_CONTROL                      (0x06 << 1)
        #define DIGITAL_AUDIO_INTERFACE_FORMAT          (0x07 << 1)
        #define SAMPLE_RATE_CONTROL                     (0x08 << 1)
        #define DIGITAL_INTERFACE_ACTIVATION            (0x09 << 1)
        #define RESET_REGISTER                          (0x0F << 1)
        
        #define CLOCK_NORMAL        0
        #define CLOCK_USB           1
        #define MCLK                0
        #define MCLK_DIV2           1
};

#endif


