/**
 ******************************************************************************
 * @file    pololu5mag.cpp
 * @author  AST / EST
 * @version V0.0.1
 * @date    08-October-2014
 * @brief   Implementation file for the Pololu5Mag singleton class
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2014 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
*/ 
    
/* Includes ------------------------------------------------------------------*/
#include "mbed.h"
#include "pololu5mag.h"

/* Static variables ----------------------------------------------------------*/
Pololu5Mag* Pololu5Mag::_instance = NULL;


/* Methods -------------------------------------------------------------------*/
/**
 * @brief  Constructor
 */
Pololu5Mag::Pololu5Mag(DevI2C *ext_i2c, DevSPI *ext_spi) : 
    dev_i2c(ext_i2c),
    dev_spi(ext_spi)
{ 
    imus = new AltIMU_10_v5* [2];
    imus[0] = AltIMU_10_v5::Instance(ext_i2c,0);
    imus[1] = AltIMU_10_v5::Instance(ext_i2c,1);
    
    magnetometers = new LIS3MDL* [NUMBER_OF_MAGNETOMETERS];
    magnetometers[0] = new LIS3MDL(*dev_spi, (M1_PIN_SPI_CS));
    magnetometers[1] = new LIS3MDL(*dev_spi, (M2_PIN_SPI_CS));
    magnetometers[2] = new LIS3MDL(*dev_spi, (M3_PIN_SPI_CS));
    magnetometers[3] = imus[0]->magnetometer;
    magnetometers[4] = imus[1]->magnetometer;
}

/**
 * @brief     Get singleton instance
 * @return    a pointer to the initialized singleton instance of class Pololu5Mag.
 *            A return value of NULL indicates an out of memory situation.
 * @param[in] ext_i2c (optional) pointer to an instance of DevI2C to be used
 *            for communication on the expansion board. 
 *            Defaults to NULL.
 *            Taken into account only on the very first call of one of the 'Instance' functions.
 *            If not provided a new DevI2C will be created with standard
 *            configuration parameters.
 *            The used DevI2C object gets saved in instance variable ext_i2c.
 * @param[in] ext_spi (optional) pointer to an instance of DevSPI to be used
 *            for communication on the expansion board. 
 *            Defaults to NULL.
 *            Taken into account only on the very first call of one of the 'Instance' functions.
 *            If not provided a new DevSPI will be created with standard
 *            configuration parameters.
 *            The used DevSPI object gets saved in instance variable dev_spi.
 */
Pololu5Mag* Pololu5Mag::Instance(DevI2C *ext_i2c, DevSPI *ext_spi) {
    if(_instance == NULL) {
        if(ext_spi == NULL)
            ext_spi = new DevSPI(F746ZG_PIN_SPI_MOSI, F746ZG_PIN_SPI_MISO, F746ZG_PIN_SPI_SCLK);
            
        if(ext_i2c == NULL)
            ext_i2c = new DevI2C(F746ZG_PIN_I2C_SDA, F746ZG_PIN_I2C_SCL);

        if(ext_i2c != NULL && ext_spi != NULL)
            _instance = new Pololu5Mag(ext_i2c, ext_spi);
    
        if(_instance != NULL) {
            bool ret = _instance->Init();
            if(!ret) {
                error("Failed to init Pololu5Mag expansion board!\n");
            }
        }
    }

    return _instance;
}

/**
 * @brief     Get singleton instance
 * @return    a pointer to the initialized singleton instance of class Pololu5Mag.
 *            A return value of NULL indicates an out of memory situation.
 * @param[in] sda I2C data line pin.
 *            Taken into account only on the very first call of one of the 'Instance' functions.
 *            A new DevI2C will be created based on parameters 'sda' and 'scl'.
 *            The used DevI2C object gets saved in instance variable dev_i2c.
 * @param[in] scl I2C clock line pin.
 *            Taken into account only on the very first call of one of the 'Instance' functions.
 *            A new DevI2C will be created based on parameters 'sda' and 'scl'.
 *            The used DevI2C object gets saved in instance variable dev_i2c.
 * @param[in] mosi SPI MOSI line pin.
 *            Taken into account only on the very first call of one of the 'Instance' functions.
 *            A new DevSPI will be created based on parameters 'mosi', 'miso' and 'sclk'.
 *            The used DevSPI object gets saved in instance variable dev_spi.
 * @param[in] miso SPI MISO line pin.
 *            Taken into account only on the very first call of one of the 'Instance' functions.
 *            A new DevSPI will be created based on parameters 'mosi', 'miso' and 'sclk'.
 *            The used DevSPI object gets saved in instance variable dev_spi.
 * @param[in] sclk SPI clock line pin.
 *            Taken into account only on the very first call of one of the 'Instance' functions.
 *            A new DevSPI will be created based on parameters 'mosi', 'miso' and 'sclk'.
 *            The used DevSPI object gets saved in instance variable dev_spi.
 */
Pololu5Mag* Pololu5Mag::Instance(PinName sda, PinName scl, PinName mosi, PinName miso, PinName sclk) {
    if(_instance == NULL) {
        DevI2C *ext_i2c = new DevI2C(sda, scl);
        DevSPI *ext_spi = new DevSPI(mosi, miso, sclk);

        if(ext_i2c != NULL && ext_spi != NULL)
            _instance = new Pololu5Mag(ext_i2c, ext_spi);
    
        if(_instance != NULL) {
            bool ret = _instance->Init();
            if(!ret) {
                error("Failed to init Pololu5Mag expansion board!\n");
            }
        }
    }

    return _instance;
}


/**
 * @brief  Initialize the singleton's magnetometers
 * @retval true if initialization successful, 
 * @retval false otherwise
 */
bool Pololu5Mag::Init_LIS3MDL(void) {
    uint8_t m_id = 0;
    MAGNETO_InitTypeDef InitStructure;
    
    /* Configure sensor */
    InitStructure.M_FullScale = 4.0f;
    InitStructure.M_OperatingMode = LIS3MDL_M_MD_CONTINUOUS;
    InitStructure.M_XYOperativeMode = LIS3MDL_M_OM_HP;
    InitStructure.M_OutputDataRate = 1000.0f;

    dev_spi->frequency(1e2);
    /* Check presence */
    for(int i = 0; i < NUMBER_OF_MAGNETOMETERS-2; i++){
        if((magnetometers[i]->read_id(&m_id) != MAGNETO_OK) ||
            (m_id != I_AM_LIS3MDL_M))
        {
            printf(" m%d_id: %d, SHOULDBE: %d", i+1, m_id, I_AM_LIS3MDL_M);
        }

        if(magnetometers[i]->init(&InitStructure) != MAGNETO_OK)
        {
            printf("%d init fail",i);
            return false;
        }
    }
      
    return true;
}

int Pololu5Mag::get_m_axes_raw(int16_t pData[][3])
{
    for(int i = 0; i < NUMBER_OF_MAGNETOMETERS; i++)
    {
        if(magnetometers[i]->get_m_axes_raw((int16_t *)pData[i]) != 0)
        {
            return -1;
        }
    }
    return 0;
}
int Pololu5Mag::get_m_axes(int32_t **pData)
{
    for(int i = 0; i < NUMBER_OF_MAGNETOMETERS; i++)
    {
        if(magnetometers[i]->get_m_axes(pData[i]) != 0)
        {
            return -1;
        }
    }
    return 0;
}


int Pololu5Mag::get_m_axes_raw(int magIndex, int16_t *pData)
{
    return magnetometers[magIndex]->get_m_axes_raw(pData);
}
bool Pololu5Mag::resetMagnetometer(int magIndex)
{
    if(magnetometers[magIndex]->resetRegs() != MAGNETO_OK)
    {
        return MAGNETO_ERROR;
    }
    
    uint8_t m_id = 0;
    MAGNETO_InitTypeDef InitStructure;
    
    /* Configure sensor */
    InitStructure.M_FullScale = 4.0f;
    InitStructure.M_OperatingMode = LIS3MDL_M_MD_CONTINUOUS;
    InitStructure.M_XYOperativeMode = LIS3MDL_M_OM_HP;
    InitStructure.M_OutputDataRate = 1000.0f;

    if(magIndex < 3) dev_spi->frequency(1e2);
    /* Check presence */
    if((magnetometers[magIndex]->read_id(&m_id) != MAGNETO_OK) ||
        (m_id != I_AM_LIS3MDL_M))
    {
        printf(" m%d_id: %d, SHOULDBE: %d", magIndex+1, m_id, I_AM_LIS3MDL_M);
    }

    if(magnetometers[magIndex]->init(&InitStructure) != MAGNETO_OK)
    {
        printf("%d init fail",magIndex);
        return false;
    }
      
    return true;
}
    