/**
  ******************************************************************************
  * @file    pololu5mag.h
  * @author  AST / EST
  * @version V0.0.1
  * @date    14-April-2015
  * @brief   This header file is intended to manage the differences between 
  *          the different supported base-boards which might mount the
  *          ALTIMU_10_V5 MEMS Inertial & Environmental Nucleo Expansion Board.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */ 

/* Define to prevent from recursive inclusion --------------------------------*/
#ifndef __POLOLU5MAG_H
#define __POLOLU5MAG_H

/* Includes ------------------------------------------------------------------*/
#include "mbed.h"
#include "pololu5mag_targets.h"
#include "lis3mdl/lis3mdl_class.h"
#include "AltIMU_10_v5/altimu_10_v5.h"

/* Classes -------------------------------------------------------------------*/
/** Class Pololu5Mag is intended to represent the MEMS Inertial & Environmental 
 *  Nucleo Expansion Board with the same name.
 *
 *  The expansion board is featuring basically five IPs:\n\
 *  -# a LIS3MDL 3-Axis Magnetometer\n
 *  -# and a AltIMU-10 v5 imu\n 
 *
 * It is intentionally implemented as a singleton because only one
 * Pololu5Mag at a time might be deployed in a HW component stack.\n
 * In order to get the singleton instance you have to call class method `Instance()`, 
 * e.g.:
 * @code
 * // Inertial & Environmental expansion board singleton instance
 * static Pololu5Mag *<TODO>_expansion_board = Pololu5Mag::Instance();
 * @endcode
 */
class Pololu5Mag
{
 protected:
    Pololu5Mag(DevI2C *ext_i2c, DevSPI *ext_spi);

    ~Pololu5Mag(void) {
        /* should never be called */
        error("Trial to delete Pololu5Mag singleton!\n");
    }

    /**
     * @brief  Initialize the singleton's sensors to default settings
     * @retval true if initialization successful, 
     * @retval false otherwise
     */
    bool Init(void) {
        return (Init_LIS3MDL());
    }

    bool Init_LIS3MDL(void);

 public:
    static Pololu5Mag* Instance(DevI2C *ext_i2c = NULL, DevSPI *ext_spi = NULL);
    static Pololu5Mag* Instance(PinName sda, PinName scl, PinName mosi, PinName miso, PinName sclk);

    DevI2C  *dev_i2c;
    DevSPI  *dev_spi;

    LIS3MDL **magnetometers;
    AltIMU_10_v5 **imus;
    
    void setI2cFrequency(int hz){ dev_i2c->frequency(hz); }
    void setSpiFrequency(int hz){ dev_spi->frequency(hz); }
    int get_m_axes_raw(int16_t pData[][3]);
    int get_m_axes_raw(int magIndex, int16_t *pData);
    int get_m_axes(int32_t **pData);
    bool resetMagnetometer(int magIndex);
    
    MagneticSensor* magneticArray(int i){
        if(i >= 0 && i < NUMBER_OF_MAGNETOMETERS) return magnetometers[i];
        return NULL;
    }
    
    GyroSensor* gyroArray(int i){
        if(i >= 0 && i < 2) return imus[i]->GetGyroscope();
        return NULL;
    }
    
    MotionSensor* motionArray(int i){
        if(i >= 0 && i < 2) return imus[i]->GetAccelerometer();
        return NULL;
    }
    
    PressureSensor* pressureArray(int i){
        if(i >= 0 && i < 2) return imus[i]->pt_sensor;
        return NULL;
    }
    
    TempSensor* tempArray(int i){
        if(i >= 0 && i < 2) return imus[i]->pt_sensor;
        return NULL;
    }

 private:
    static Pololu5Mag *_instance;
};

#endif /* __POLOLU5MAG_H */