#include "motor.h"

Motor::Motor(PinName p, PinName d, char* n):
    _name(n),
    _resolution(200),
    _phase(2),
    _maxRotateSpeed(4000),
    _minDelay(2.5),
    _pulse(p),
    _dir(d),
    _pulseCounter(0),
    _rpm(0),
    _cw(0),
    _ccw(1),
    _rotateDirection(CCW),
    _lock(false),
    _rotateFinishEvent(NULL),
    _rotateCheckEvent(NULL)
{
    _pulse = 0;
    _dir = 0;
    setSpeed(1);
}

void Motor::toggleOn(void)
{
    _pulse = 1;
    if(_pulseCounter > _goalPulseCount && _rotateDirection == CCW)
    {
        _rotateDirection = CW;
        _dir = _cw;
    }
    else if(_pulseCounter < _goalPulseCount && _rotateDirection == CW)
    {
        _rotateDirection = CCW;
        _dir = _ccw;
    }
    _pulseCounter += (_rotateDirection==CCW?1:-1);
    
    if( _pulseCounter != _goalPulseCount)
    {
        bool check = true;
        if(_rotateCheckEvent) check = _rotateCheckEvent();
        if(check){
            _pulseTimeout.attach_us(callback(this, &Motor::toggleOff), _onDelay);
            return;
        }
    }
    stop();
}

void Motor::toggleOff(void)
{
    _pulse = 0;
    _pulseTimeout.attach_us(callback(this, &Motor::toggleOn), _offDelay);
}

void Motor::stop(void)
{
    _pulseTimeout.detach();
    _lock = false;
    if(_rotateFinishEvent) _rotateFinishEvent();
    _rotateFinishEvent = NULL;
    _rotateCheckEvent = NULL;
}
void Motor::rotate(Direction direction, int pulseCount)
{
    if(_lock || (_rpm == 0)) {
        return;
    }
    _lock = true;
    pulseCount = (pulseCount > 0 ? pulseCount : -pulseCount);
    _pulseTimeout.detach();
    _pulse = 0;
    _rotateDirection = direction;
    _dir = (_rotateDirection==CCW?_ccw:_cw);
    _goalPulseCount = _pulseCounter + (_rotateDirection==CCW?pulseCount:-pulseCount);
    toggleOn();
}

int Motor::setSpeed(int rpm)
{
    if(_lock) {
        return _rpm;
    }
    _lock = true;
    _rpm = (rpm>_maxRotateSpeed?_maxRotateSpeed:(_rpm<0?0:rpm));
    if(_rpm == 0){
        _lock = false;
        return _rpm;
    }
    int frequency = _rpm*_phase*_resolution/60;
    float T = 1000000.0f/(float)frequency; // in us
    if(T>_minDelay*2.0f){
        _onDelay=(int)(T/2.0f);
        _offDelay=_onDelay;
    }
    else
    {
        _onDelay = (int)_minDelay+((_minDelay-(int)_minDelay)>0?1:0);
        _offDelay = _onDelay;
        T = _minDelay*2.0f;
        frequency = (int)(1000000.0f/T);
        _rpm = frequency*60/_resolution/_phase;
    }
    _lock = false;
    return _rpm;
}

//void Motor::rotateCallBack(void)
//{
//    _pulseTimeout.detach();
//    _pos = _goal;
//    _lock = false;
//    _goal = 0;
//}

CD_2D34M::CD_2D34M(char* n, PinName d, PinName p, PinName a, PinName r):
    Motor(p,d,n), 
    _alarm(a), 
    _reset(r)
{
    _resolution = (1600);
    _phase = (2);
    _maxRotateSpeed = (4000);
    _minDelay = (2.5f);
    set_cw_signal(CD_2D34M__CW);
    _reset = 1;
#ifdef ALARM
    _alarm.fall(this, &CD_2D34M::alarm_event);
#endif
    //alarm.fall(this, &CD_2D34M::alarm_reset);
}
void CD_2D34M::alarm_event()
{
    _reset = 0;
    wait(0.01);
    alarm_reset();
}
void CD_2D34M::alarm_reset()
{
    _reset = 1;
    wait(0.01);
}

CD_2D24MB::CD_2D24MB(char* n, PinName p, PinName d, PinName m):
    Motor(p,d,n),
    _mf(m)
{
    _resolution = (4000);
    _phase = (2);
    _maxRotateSpeed = (4000);
    _minDelay = (2.5f);
    set_cw_signal(CD_2D24MB__CW);
    _mf = 1;
}

AlphaSmart::AlphaSmart(char* n, PinName pulse, PinName dir, PinName sv_off, PinName rst, PinName inp, PinName rdy, PinName alm, PinName ea, PinName eb, PinName ez, int res):
    Motor(pulse,dir,n),
    _sv_off(sv_off), 
    _rst(rst), 
    _inp(inp), 
    _rdy(rdy), 
    _alm(alm)
{
    _resolution = (res);
    _phase = (2);
    _maxRotateSpeed = (4000);
    _minDelay = (2.5f);
    _sv_off = 1;
    _rst = 1;
}


