/**
 ******************************************************************************
 * @file    mag_ring.h
 * @author  AST / EST
 * @version V0.0.1
 * @date    13-April-2015
 * @brief   Header file for class mag_ring representing a mag_ring
 *          expansion board
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */

/* Define to prevent from recursive inclusion --------------------------------*/
#ifndef __MAG_RING_H
#define __MAG_RING_H

/* Includes ------------------------------------------------------------------*/
#include "mbed.h"
#include "mag_ring_targets.h"
#include "lis3mdl/lis3mdl_class.h"
#include "DevSPI.h"

/* Macros -------------------------------------------------------------------*/
#define CALL_METH(obj, meth, param, ret) ((obj == NULL) ?		\
					  ((*(param) = (ret)), 0) :	\
					  ((obj)->meth(param))		\
					  )

/* Classes -------------------------------------------------------------------*/
/** Class mag_ring is intended to represent the MEMS Inertial & Environmental 
 *  Nucleo Expansion Board with the same name.
 *
 *  The expansion board is featuring basically four IPs:\n
 *  -# a HTS221 Relative Humidity and Temperature Sensor\n
 *  -# a LIS3MDL 3-Axis Magnetometer\n
 *  -# a LPS25H MEMS Pressure Sensor (and Temperature Sensor)\n
 *  -# and a LSM6DS33 3D Acceleromenter and 3D Gyroscope\n
 *
 * The expansion board features also a DIL 24-pin socket which makes it possible
 * to add further MEMS adapters and other sensors (e.g. UV index). 
 *
 * It is intentionally implemented as a singleton because only one
 * mag_ring at a time might be deployed in a HW component stack.\n
 * In order to get the singleton instance you have to call class method `Instance()`, 
 * e.g.:
 * @code
 * // Inertial & Environmental expansion board singleton instance
 * static mag_ring *<TODO>_expansion_board = mag_ring::Instance();
 * @endcode
 */
class MagRing
{
 protected:
	MagRing(DevSPI *ext_spi);

	~MagRing(void) {
	    /* should never be called */
	    error("Trial to delete mag_ring singleton!\n");
	}

	/**
	 * @brief  Initialize the singleton's sensors to default settings
	 * @retval true if initialization successful, 
	 * @retval false otherwise
	 */
	bool Init(void) {
		return (Init_LIS3MDL());
	}

	bool Init_LIS3MDL(void);

 public:
	static MagRing* Instance(DevSPI *ext_spi = NULL);
	static MagRing* Instance(PinName mosi, PinName miso, PinName sclk);

	DevSPI  *dev_spi;

	LIS3MDL **magnetometers;
/*	GyroSensor *GetGyroscope(void) {
		return gyro_lsm6ds33;
	}
	MotionSensor *GetAccelerometer(void) {
		return gyro_lsm6ds33;
	}*/
	void setSpiFrequency(int hz){ dev_spi->frequency(hz); }
	int get_m_axes_raw(int16_t pData[][3]);
	int get_m_axes_raw(int magIndex, int16_t *pData);
	int get_m_axes(int32_t **pData);
	bool resetMagnetometer(int magIndex);
//	void setMagenticRegister(float **);
//	LSM6DS33 *gyro_lsm6ds33;

 private:
	static MagRing *_instance;
};

#endif /* __MAG_RING_H */

