/**
 ******************************************************************************
 * @file    mag_ring.cpp
 * @author  AST / EST
 * @version V0.0.1
 * @date    08-October-2014
 * @brief   Implementation file for the MagRing singleton class
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2014 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
*/ 
    
/* Includes ------------------------------------------------------------------*/
#include "mbed.h"
#include "mag_ring.h"

/* Static variables ----------------------------------------------------------*/
MagRing* MagRing::_instance = NULL;


/* Methods -------------------------------------------------------------------*/
/**
 * @brief  Constructor
 */
MagRing::MagRing(DevSPI *ext_spi) : dev_spi(ext_spi)
{ 
	magnetometers = new LIS3MDL* [NUMBER_OF_MAGNETOMETERS];
	magnetometers[0] = new LIS3MDL(*dev_spi, (M1_PIN_SPI_CS));
	magnetometers[1] = new LIS3MDL(*dev_spi, (M2_PIN_SPI_CS));
	magnetometers[2] = new LIS3MDL(*dev_spi, (M3_PIN_SPI_CS));
	magnetometers[3] = new LIS3MDL(*dev_spi, (M4_PIN_SPI_CS));
	magnetometers[4] = new LIS3MDL(*dev_spi, (M5_PIN_SPI_CS));
	magnetometers[5] = new LIS3MDL(*dev_spi, (M6_PIN_SPI_CS));
	magnetometers[6] = new LIS3MDL(*dev_spi, (M7_PIN_SPI_CS));
	magnetometers[7] = new LIS3MDL(*dev_spi, (M8_PIN_SPI_CS));
	magnetometers[8] = new LIS3MDL(*dev_spi, (M1_PIN_SPI_CS_1));
	magnetometers[9] = new LIS3MDL(*dev_spi, (M2_PIN_SPI_CS_1));
	magnetometers[10] = new LIS3MDL(*dev_spi, (M3_PIN_SPI_CS_1));
	magnetometers[11] = new LIS3MDL(*dev_spi, (M4_PIN_SPI_CS_1));
	magnetometers[12] = new LIS3MDL(*dev_spi, (M5_PIN_SPI_CS_1));
	magnetometers[13] = new LIS3MDL(*dev_spi, (M6_PIN_SPI_CS_1));
	magnetometers[14] = new LIS3MDL(*dev_spi, (M7_PIN_SPI_CS_1));
	magnetometers[15] = new LIS3MDL(*dev_spi, (M8_PIN_SPI_CS_1));
}

/**
 * @brief     Get singleton instance
 * @return    a pointer to the initialized singleton instance of class MagRing.
 *            A return value of NULL indicates an out of memory situation.
 * @param[in] ext_spi (optional) pointer to an instance of DevSPI to be used
 *            for communication on the expansion board. 
 *            Defaults to NULL.
 *            Taken into account only on the very first call of one of the 'Instance' functions.
 *            If not provided a new DevSPI will be created with standard
 *            configuration parameters.
 *            The used DevSPI object gets saved in instance variable dev_spi.
 * @param[in] ff_irq_pin (optional) PinName of the pin associated to asynchronous 
 *            (i.e. interrupt based) free fall detection in case a LSM6DS33 3D 
 *            Acceleromenter and 3D Gyroscope is mounted on top of the DIL 24-pin socket.
 *            Defaults to F746ZG_PIN_FF.
 *            Taken into account only on the very first call of one of the 'Instance' functions.
 *            A value of 'NC' will avoid instantiation of the LSM6DS33 even if present.
 */
MagRing* MagRing::Instance(DevSPI *ext_spi) {
	if(_instance == NULL) {
		if(ext_spi == NULL)
			ext_spi = new DevSPI(F746ZG_PIN_SPI_MOSI, F746ZG_PIN_SPI_MISO, F746ZG_PIN_SPI_SCLK);

		if(ext_spi != NULL)
			_instance = new MagRing(ext_spi);
	
		if(_instance != NULL) {
			bool ret = _instance->Init();
			if(!ret) {
				error("Failed to init MagRing expansion board!\n");
			}
		}
	}

	return _instance;
}

/**
 * @brief     Get singleton instance
 * @return    a pointer to the initialized singleton instance of class MagRing.
 *            A return value of NULL indicates an out of memory situation.
 * @param[in] sda SPI data line pin.
 *            Taken into account only on the very first call of one of the 'Instance' functions.
 *            A new DevSPI will be created based on parameters 'sda' and 'scl'.
 *            The used DevSPI object gets saved in instance variable dev_spi.
 * @param[in] scl SPI clock line pin.
 *            Taken into account only on the very first call of one of the 'Instance' functions.
 *            A new DevSPI will be created based on parameters 'sda' and 'scl'.
 *            The used DevSPI object gets saved in instance variable dev_spi.
 * @param[in] ff_irq_pin (optional) PinName of the pin associated to asynchronous 
 *            (i.e. interrupt based) free fall detection in case a LSM6DS33 3D 
 *            Acceleromenter and 3D Gyroscope is mounted on top of the DIL 24-pin socket.
 *            Defaults to NC.
 *            Taken into account only on the very first call of one of the 'Instance' functions.
 *            A value of 'NC' will avoid instantiation of the LSM6DS33 even if present.
 */
MagRing* MagRing::Instance(PinName mosi, PinName miso, PinName sclk) {
	if(_instance == NULL) {
		DevSPI *ext_spi = new DevSPI(mosi, miso, sclk);

		if(ext_spi != NULL)
			_instance = new MagRing(ext_spi);
	
		if(_instance != NULL) {
			bool ret = _instance->Init();
			if(!ret) {
				error("Failed to init MagRing expansion board!\n");
			}
		}
	}

	return _instance;
}


/**
 * @brief  Initialize the singleton's magnetometers
 * @retval true if initialization successful, 
 * @retval false otherwise
 */
bool MagRing::Init_LIS3MDL(void) {
	uint8_t m_id = 0;
	MAGNETO_InitTypeDef InitStructure;
	
	/* Configure sensor */
	InitStructure.M_FullScale = 4.0f;
	InitStructure.M_OperatingMode = LIS3MDL_M_MD_CONTINUOUS;
	InitStructure.M_XYOperativeMode = LIS3MDL_M_OM_UHP;
	InitStructure.M_OutputDataRate = 100.0f;

	dev_spi->frequency(1e2);
	/* Check presence */
	for(int i = 0; i < NUMBER_OF_MAGNETOMETERS; i++){
		if((magnetometers[i]->read_id(&m_id) != MAGNETO_OK) ||
	   		(m_id != I_AM_LIS3MDL_M))
		{
			printf(" m%d_id: %d, SHOULDBE: %d", i+1, m_id, I_AM_LIS3MDL_M);/*
			for(int j = 0; j < NUMBER_OF_MAGNETOMETERS; j++){
				delete magnetometers[j];
				magnetometers[j] = NULL;
			}
			delete [] magnetometers;
			magnetometers = NULL;
			return true;*/
		}

		if(magnetometers[i]->init(&InitStructure) != MAGNETO_OK)
		{
			printf("%d init fail",i);
			return false;
		}
    }
      
	return true;
}

int MagRing::get_m_axes_raw(int16_t pData[][3])
{
	for(int i = 0; i < NUMBER_OF_MAGNETOMETERS; i++)
	{
		if(magnetometers[i]->get_m_axes_raw((int16_t *)pData[i]) != 0)
		{
			return -1;
		}
	}
	return 0;
}
int MagRing::get_m_axes(int32_t **pData)
{
	for(int i = 0; i < NUMBER_OF_MAGNETOMETERS; i++)
	{
		if(magnetometers[i]->get_m_axes(pData[i]) != 0)
		{
			return -1;
		}
	}
	return 0;
}


int MagRing::get_m_axes_raw(int magIndex, int16_t *pData)
{
	return magnetometers[magIndex]->get_m_axes_raw(pData);
}
bool MagRing::resetMagnetometer(int magIndex)
{
	if(magnetometers[magIndex]->resetRegs() != MAGNETO_OK)
	{
		return MAGNETO_ERROR;
	}
	
	uint8_t m_id = 0;
	MAGNETO_InitTypeDef InitStructure;
	
	/* Configure sensor */
	InitStructure.M_FullScale = 4.0f;
	InitStructure.M_OperatingMode = LIS3MDL_M_MD_CONTINUOUS;
	InitStructure.M_XYOperativeMode = LIS3MDL_M_OM_UHP;
	InitStructure.M_OutputDataRate = 100.0f;

	dev_spi->frequency(1e2);
	/* Check presence */
		if((magnetometers[magIndex]->read_id(&m_id) != MAGNETO_OK) ||
	   		(m_id != I_AM_LIS3MDL_M))
		{
			printf(" m%d_id: %d, SHOULDBE: %d", magIndex+1, m_id, I_AM_LIS3MDL_M);/*
			for(int j = 0; j < NUMBER_OF_MAGNETOMETERS; j++){
				delete magnetometers[j];
				magnetometers[j] = NULL;
			}
			delete [] magnetometers;
			magnetometers = NULL;
			return true;*/
		}

		if(magnetometers[magIndex]->init(&InitStructure) != MAGNETO_OK)
		{
			printf("%d init fail",magIndex);
			return false;
		}
      
	return true;
}
	
