/**
 ******************************************************************************
 * @file    altimu_10_v5.h
 * @author  AST / EST
 * @version V0.0.1
 * @date    13-April-2015
 * @brief   Header file for class AltIMU_10_v5 representing a AltIMU-10 v5
 *          expansion board
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */

/* Define to prevent from recursive inclusion --------------------------------*/
#ifndef __ALTIMU_10_V5_H
#define __ALTIMU_10_V5_H

/* Includes ------------------------------------------------------------------*/
#include "mbed.h"
#include "altimu_10_v5_targets.h"
#include "lis3mdl/lis3mdl_class.h"
#include "lps25h/lps25h_class.h"
#include "lsm6ds33/lsm6ds33_class.h"
#include "DevI2C.h"

/* Macros -------------------------------------------------------------------*/
#define CALL_METH(obj, meth, param, ret) ((obj == NULL) ?		\
					  ((*(param) = (ret)), 0) :	\
					  ((obj)->meth(param))		\
					  )

/* Classes -------------------------------------------------------------------*/
/** Class AltIMU_10_v5 is intended to represent the MEMS Inertial & Environmental 
 *  Nucleo Expansion Board with the same name.
 *
 *  The expansion board is featuring basically four IPs:\n
 *  -# a HTS221 Relative Humidity and Temperature Sensor\n
 *  -# a LIS3MDL 3-Axis Magnetometer\n
 *  -# a LPS25H MEMS Pressure Sensor (and Temperature Sensor)\n
 *  -# and a LSM6DS33 3D Acceleromenter and 3D Gyroscope\n
 *
 * The expansion board features also a DIL 24-pin socket which makes it possible
 * to add further MEMS adapters and other sensors (e.g. UV index). 
 *
 * It is intentionally implemented as a singleton because only one
 * AltIMU_10_v5 at a time might be deployed in a HW component stack.\n
 * In order to get the singleton instance you have to call class method `Instance()`, 
 * e.g.:
 * @code
 * // Inertial & Environmental expansion board singleton instance
 * static AltIMU_10_v5 *<TODO>_expansion_board = AltIMU_10_v5::Instance();
 * @endcode
 */
class AltIMU_10_v5
{
 protected:
	AltIMU_10_v5(DevI2C *ext_i2c, uint8_t SA0=1);

	~AltIMU_10_v5(void) {
	    /* should never be called */
	    error("Trial to delete AltIMU_10_v5 singleton!\n");
	}

	/**
	 * @brief  Initialize the singleton's sensors to default settings
	 * @retval true if initialization successful, 
	 * @retval false otherwise
	 */
	bool Init(void) {
		return (Init_LIS3MDL() &&
			Init_LPS25H() &&
			Init_LSM6DS33());
	}

	bool Init_LIS3MDL(void);
	bool Init_LPS25H(void);
	bool Init_LSM6DS33(void);

 public:
	static AltIMU_10_v5* Instance(DevI2C *ext_i2c = NULL, uint8_t SA0 = 1);
	static AltIMU_10_v5* Instance(PinName sda, PinName scl, uint8_t SA0 = 1);

	DevI2C  *dev_i2c;

	LIS3MDL *magnetometer;
	LPS25H  *pt_sensor;

	GyroSensor *GetGyroscope(void) {
		return gyro_lsm6ds33;
	}
	MotionSensor *GetAccelerometer(void) {
		return gyro_lsm6ds33;
	}
	LSM6DS33 *gyro_lsm6ds33;
	
	
	void setI2cFrequency(int hz){ dev_i2c->frequency(hz); }

 private:
	static AltIMU_10_v5 *_instance0;
	static AltIMU_10_v5 *_instance1;
};

#endif /* __ALTIMU_10_V5_H */
