#ifndef RCS620S_AB_H
#define RCS620S_AB_H

#include "mbed.h"

//コマンドは別ファイルへ保存
#include "RCS620S_command.h"

/**
  @def READ_BUF_SIZE
  受信バッファのバイト数 (Number of bytes in the receive buffer)
*/
#define READ_BUF_SIZE 256

/**
 @def UART_TRANSMIT_FAIL_TIMEOUT_MS
 UARTコマンド送信後ACK応答が返ってこない場合にタイムアウトする時間(ミリ秒)
 (Time-out when ACK response is not returned after sending UART command.)
*/
#define UART_TRANSMIT_FAIL_TIMEOUT_MS 1000

/** RCS620S_AB class.
* RC-S620/Sをイニシエータとして使用しNFC Type-A/Bカードと通信するためのライブラリです。
* (This is library to communicate with NFC Type-A/B card by using RC-S620/S as an initiator.)
*/
class RCS620S_AB{
public:
    /** 
     * @brief クラス内受信バッファのバイト数です。(The number of bytes in the receive buffer within the class.)
     *
     * マクロで定義した READ_BUF_SIZE の値が代入されています。クラス呼び出し元でバッファ配列初期化時の要素数定義にご利用ください。
     * (The value of READ_BUF_SIZE defined by the macro is assigned.
     * Please use it to define the number of elements when the buffer array is initialized by the class caller.)
     */
    static const int RETURN_ENVELOPE_SIZE = READ_BUF_SIZE;
    
    /** 
     * @brief ボーレートは115200固定です(RC-S620/S デフォルト値)。(The baud rate is fixed to 115200 (RC-S620/S default value).)
     * @param tx UART TX pin
     * @param rx UART RX pin
     */
    RCS620S_AB(PinName tx, PinName rx);

/***************/
/**** 制御系 ****/
/***************/
    
    /** 
     * @brief パワーダウンコマンドを送信します。
     * (Send powerDown command.)
     *
     * wakeup条件はUARTを指定しています。変更される場合、RCS620S_command.h ファイル内の wired_data_RFConfiguration_PowerDown[] を変更ください。
     *
     * (The wakeup condition specifies UART.
     * If you want to change it, please change "wired_data_RFConfiguration_PowerDown[]" in "RCS620S_command.h".)
     */
    void powerDown();
    
    /** 
     * @brief ウェイクアップコマンド(0x55)を送信します(Send wakeUp command(0x55).)
     */
    void wakeUp();
    
    /** 
     * @brief RF停止コマンド送信後、リセットコマンドを送信します。
     * (Send not only Reset command but also RFoff command.)
     * 
     * RF停止は、"カードを"リセットするために送信します。
     *
     * (RFoff is used to reset a "card".)
     */
    void reset();
    
/***************/
/**** 受信系 ****/
/***************/

    /** 
    * @brief 受信バッファ中の全データ(ACK～レスポンスのポストアンブルまで)を取得します。主としてデバッグ用です。
    * (Get all data (ACK to response postamble) in the receive buffer. Mainly for debugging use.)
    * @param ansArray[] データを返すための配列へのポインタ。サイズの初期化には RETURN_ENVELOPE_SIZE をご利用ください。
    * (A pointer to an array to return the data. Please use RETURN_ENVELOPE_SIZE to initialize the size.)
    * @param *ansLen データ長(バイト数)を返すための変数へのポインタ。(Pointer to a variable to return the data length(number of bytes).)
    */
    void getAllRes(uint8_t ansArray[], int *ansLen);
    
    /** 
    * @brief 受信バッファからカードリーダのレスポンスデータを取得します。
    * (Get the response data of the card reader from the receive buffer.)
    * @param ansArray[] データを返すための配列へのポインタ。サイズの初期化には RETURN_ENVELOPE_SIZE をご利用ください。
    * (A pointer to an array to return the data. Please use RETURN_ENVELOPE_SIZE to initialize the size.)
    * @param *ansLen データ長(バイト数)を返すための変数へのポインタ。(Pointer to a variable to return the data length(number of bytes).)
    */
    void getRes(uint8_t ansArray[], int *ansLen);
    
    /** 
    * @brief inListPasseveTarget 及び inDataExchange レスポンスからカードからのレスポンス部だけを取得します。前記2つのコマンド以外のレスポンスに対しては使用できません。
    * (Get only the response part from the card is acquired from the inListPasseveTarget and inDataExchange responses.
    * It cannot be used for responses other than the above two commands.)
    * @param ansArray[] データを返すための配列へのポインタ。サイズの初期化には RETURN_ENVELOPE_SIZE をご利用ください。
    * (A pointer to an array to return the data. Please use RETURN_ENVELOPE_SIZE to initialize the size.)
    * @param *ansLen データ長(バイト数)を返すための変数へのポインタ。(Pointer to a variable to return the data length(number of bytes).)
    */
    void getCardRes(uint8_t ansArray[], int *ansLen);

/***************/
/**** 送信系 ****/
/***************/
 
    /** 
    * @brief カードリーダへコマンドを送信し、レスポンスを受信します。受信完了までwait()します。本メソッド実行後にgetRse()でレスポンスデータを取出してください。
    * (This method sends commands to the card reader and receives responses. Wait() until the reception is completed.
    * After executing this method, please use the getRse() to retrieve the response data.)
    *
    * 【重要】送受信中に、割り込み処理や別の重い処理が発生しないようご配慮ください。
    * レスポンスを受信完了するまでwait()で処理を停止しますが、割り込みで別の重い処理が走るとデータを取りこぼします。
    * ただし、レスポンスの受信にUART受信割込みを使用しているため、割り込み禁止を使用する際はuartまで止めないようご注意ください。
    *
    * (IMPORTANT! Please take care not to cause interrupt processing or other heavy processing during transmission and reception.
    * The process is stopped by wait () until the response is received, but the data is dropped when another heavy process runs by the interrupt.
    * However, since the UART reception interrupt is used to receive the response, please be careful not to stop until uart when using the interrupt disable.)
    *
    * @param wired_packet_data[] コマンドのデータ部(0xD4～DCSの直前まで)
    * (Command data part (From 0xD4 to just before DCS))
    * @param wired_packet_data_len コマンドのデータ長(バイト数)。 sizeof(command array) で与えてください。
    * (Data length(number of bytes) of command.Give it as sizeof (command array).)
    */
    void sendCommand_and_RecieveRes(const uint8_t wired_packet_data[], uint16_t wired_packet_data_len);

    /** 
    * @brief InDataExchangeを使ってICカードへNFCコマンドを送信し、レスポンスを受信します。内部でsendCommand_and_RecieveRes()を呼び出しています。
    * 本メソッド実行後にgetCardRes()またはgetRse()でレスポンスデータを取出してください。
    * (NFC command is sent to the IC card via InDataExchange and the response is received.
    * This method internally calls sendCommand_and_RecieveRes().
    * After executing this method, please use getCardRes() or getRse() to retrieve the response data.)
    * 
    * ※inDataExchange以外の方法でカードと通信したい場合、NFCコマンドを元に使いたいコマンドを組み立ててsendCommand_and_RecieveRes()で送信してください。
    *
    * (If you want to communicate with the card by a method other than inDataExchange,
    *  assemble the command you want to use based on the NFC command and send it with sendCommand_and_RecieveRes().)
    *
    * @param card_command[] NFCコマンド(NFC command)
    * @param card_command_len NFCコマンドのデータ長(バイト数)。 sizeof(command array) で与えてください。
    * (Data length(number of bytes) of NFC command.Give it as sizeof (command array).)
    */
    void sendInDataExchange_and_RecieveRes(const uint8_t card_command[], uint16_t card_command_len);

    /** 
    * @brief inListPassiveTargetを使ってNFC-TypeAをポーリングし、レスポンスを受信します。内部でsendCommand_and_RecieveRes()を呼び出しています。
    * 本メソッド実行後にgetCardRes()またはgetRse()でレスポンスデータを取出してください。
    * (This method polls NFC-TypeA via inListPassiveTarget and receives the response. This method internally calls sendCommand_and_RecieveRes().
    * After executing this method, please use getCardRes() or getRse() to retrieve the response data.)
    *
    * inListPassiveTargetのパラメータを変更したい場合は、"RCS620S_command.h"内の"wired_data_InListPassiveTarget_typeA[]"を書き換えてください。
    * タイムアウト等の設定値はRC-S620/Sデフォルトのままです。必要に応じ別途RFConfigurationコマンドで設定してください。
    *
    * (If you want to change the parameters of inListPassiveTarget, rewrite "wired_data_InListPassiveTarget_typeA []" in "RCS620S_command.h".
    * The set values such as timeout are the default of RC-S620/S. Please set it separately with the RF Configuration command if necessary.)
    */
    void sendInListPasseveTarget_typeA();
    
    /** 
    * @brief inListPassiveTargetを使ってNFC-TypeBをポーリングし、レスポンスを受信します。内部でsendCommand_and_RecieveRes()を呼び出しています。
    * 本メソッド実行後にgetCardRes()またはgetRse()でレスポンスデータを取出してください。
    * (This method polls NFC-TypeB via inListPassiveTarget and receives the response. This method internally calls sendCommand_and_RecieveRes().
    * After executing this method, please use getCardRes() or getRse() to retrieve the response data.)
    *
    * inListPassiveTargetのパラメータを変更したい場合は、"RCS620S_command.h"内の"wired_data_InListPassiveTarget_typeB[]"を書き換えてください。
    * (AFI は 0x00 (用途を指定しない) を設定しています。)
    * 
    * タイムアウト等の設定値はRC-S620/Sデフォルトのままです。必要に応じ別途RFConfigurationコマンドで設定してください。
    *
    * (If you want to change the parameters of inListPassiveTarget, rewrite "wired_data_InListPassiveTarget_typeA []" in "RCS620S_command.h".
    * (AFI sets 0x00 (no purpose specified).)
    *
    * The set values such as timeout are the default of RC-S620/S. Please set it separately with the RF Configuration command if necessary.)
    */
    void sendInListPassiveTarget_typeB();
    
private:

    Serial _serial;
    uint8_t recieveBuffer[READ_BUF_SIZE];
    int recieveLEN;

/***************/
/**** 制御系 ****/
/***************/

    void initialize();
    
/***************/
/**** 受信系 ****/
/***************/

     //カードリーダ UART受信割込み
    void serialRxIrq();
    
    //受信割込みバッファをクリア
    void clearRecieveBuffer();
    
    //ACK応答待ち
    bool isACKrecieved(int timeout_ms);
    
    //読み取り完了まで待機
    void waitWhileRead(int cycleTime_ms);
    
    //受信バッファからrecieveBuffer[start] ～ recieveBuffer[end] を抜き出す
    void extractRes(int start, int end, uint8_t ansArray[], int *ansLen);

/***************/
/**** 検査系 ****/
/***************/

    bool checkACK();

    bool checkCardRes();
    
    //受信データをチェックしてOKならLENを返す、異常なら0を返す
    uint8_t checkRxData_and_getLEN();
};



#endif