#ifndef PWMBEEP_H
#define PWMBEEP_H

#include "mbed.h"

/** PwmBeep class.
*  PwmOutを使用してビープを鳴らすライブラリです。(A library that generate a beep using PwmOut.)
*/
class PwmBeep{
public:
    /** 
     * @param pin 使用するPwmOutポート (PwmOut port to use)
     * @param initialfreq 周波数(Hz) (Frequency(Hz))
     *
     * 周波数はsetFreq()により後で変更できます。(The frequency can be changed later with setFreq().)
     */
    PwmBeep(PinName pin, int initialfreq = 1000);
    
    /**
     * @brief ビープをオフします。(Turn off a beep.)
     */
    void turnOff();
     
     /**
     * @brief ビープをオンします。(Turn on a beep.)
     */
    void turnOn();
    
    /**
     * @brief 一度だけ指定時間鳴らします。内部でタイムアウト割り込みを使用しており、鳴らしながら他の処理ができます。ボタン操作音用です。
     * (Sounds once for a specified time. The timeout interrupt is used internally, and other processing can be done while sounding.For button touch sounds.)
     * @param time 鳴らす時間(秒) (Duration time(sec))
     */
    void oneshotOn(float time);
    
     /**
     * @brief 指定回数繰り返し鳴らします。内部でタイマー割り込みを使用しており、鳴らしながら他の処理ができます。
     * (Repeat the specified number of times. The timer interrupt is used internally, and other processing can be done while sounding.)
     * @param num 繰り返し回数 (Number of repetitions)
     * @param ontime 一周期中のON時間(秒) (On time in one cycle(sec))
     * @param offtime 一周期中のoff時間(秒) (Off time in one cycle(sec))
     *
     * ※鳴動回数でエラーコードを示すなど確実に指定回数鳴らしたい場面では、NshotOnwithWait の使用を検討ください。
     * (Consider using NshotOnwithWait in situations where you want to be sure to sound the specified number of times, 
     *  such as showing an error code by the number of times you ring.)
     */
    void NshotOn(int num,float ontime,float offtime);
     
     /**
     * @brief 指定回数繰り返し鳴らします。内部でwait()関数を使用しており、指定数鳴らし終わるまで他の処理を止めます。
     *  (Repeat the specified number of times. The wait() function is used internally, and other processing is stopped until the specified number of sounds have been played.)
     * @param num 繰り返し回数 (Number of repetitions)
     * @param ontime 一周期中のON時間(秒) (On time in one cycle(秒))
     * @param offtime 一周期中のoff時間(秒) (Off time in one cycle(秒))
     * 
     * 途中でビープを操作する割り込み処理がかからないよう、必要に応じて割り込みの禁止・許可を外部で設定してください。
     * (If necessary, disable or enable interrupts externally so that interrupt processing that operates the beep does not occur during the process.)
     */
    void NshotOnwithWait(int num,float ontime,float offtime);
    
    /**
     * @brief ビープの周波数を設定します。(Set frequency of beep.)
     * @param freq 周波数(Hz) (Frequency(Hz))
     */
    void setFreq(int freq);
 
private:
    PwmOut _pwmbeep;
    Timeout timeout;
    int counter;
    int repeat_count;
    float repeat_ontime;
    float repeat_offtime;
    
    void turnOn_and_setOffTimer();
    void turnOff_and_setOnTimer();
};

#endif