/*
* POP3 library
* Copyright (c) 2010 Hendrik Lipka
* 
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
* 
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*/

#ifndef __POP3_H__
#define __POP3_H__

#include <string>
#include <list>

#include "tcplinestream.h"

using namespace std;

/**
    The class representing a received mail message. All fields are stored in un-altered form, so character set conversion might need to be performed.
*/
class Pop3Message
{
    public:
        /**
            the UIDL this message has on the server
        */
        string id;
        /**
            the 'From:' header        
        */
        string from;
        /**
            the 'Subject:' header
        */
        string subject;
        /**
            the actual mail content, line by line
        */
        list<string> content;
        
};

class Pop3CommandResponse;

/**
    Handling all POP3 related aspects. All operations are synchronous. This class expects a set-up network connection.
    
    Needed libraries:
    * DNSResolver
    * NetServices (for TCP socket), or NetServicesMin
*/
class Pop3
{
    public:
        /**
            Creates the POP3 handler
            
            @param servername the DNS name of the server
            @param username the username
            @param password the users password
        */
        Pop3(const char* servername, const char* username, const char* password);
        
        /**
            closes the connection, and cleans up resources
        */
        ~Pop3();
        
        /**
            Retrieve a list of all message IDs. This method doesn't care whether the messages have been read already or not.
            The IDs are not sorted, but returned in the order the server delivers them.
            
            @return the list of all message IDs stored at the server.
        */
        list<string> *getMessages();
        
        /**
            retrieves as single message
            
            @param id the message ID to retrieve
            @param getSig when false, the signature will be stripped (separated by the line '-- ')
            @param deleteOnReceive when true the message will be deleted after a sucessful retrieval
            @return the message container
        */
        Pop3Message* getMessage(string id, bool getSig=false, bool deleteOnReceive=false);
        
        /**
            @param id the ID of the message to be deleted
            @return true if the deletion was sucessful
        */
        bool deleteMessage(string id);
        
        /**
            Connects to the server. Needs the ethernet connection already set up.
            
            @return true if the connection has been made (and the user logged in)
        */
        bool init();

        /**
            closes the connection, and cleans up resources
        */
        void close();
        
    private:
        Pop3CommandResponse* sendCommand(string cmd);
        Pop3CommandResponse* sendCommandMultiResponse(string cmd);
        
        TCPLineStream *_stream;
        
        const char* _username;
        const char* _password;
        const char* _servername;
        
        bool _closed;
};


#endif