/*
* mbed AR1020 library
* Copyright (c) 2010 Hendrik Lipka
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*/

#ifndef __AR1020_H_
#define __AR1020_H_

#include "mbed.h"
    
#include "touchpanel.h"
#include "touchevent.h"

/**
    class handling all the connections to a AR1020 touch screen controller
    SPI is handled by bit-banging, since the SPI-mode is non-standard. Therefore all pins can be used for the connection.
    Since the AR1020 seems to be sensitive to the changing pin signals during reset and startup, this library controls the AR1020 power (so it's Vcc must be connected to an mbed pin)   
*/    
class AR1020: public TouchPanel
{
    public:
        /**
            @params mosi the MOSI pin name (SDI)
            @params miso the MISO pin name (SDO)
            @params clk the CLK pin (SCL)
            @params enable the enable pin name (/CS)
            @params siq the SIQ pin name (SIQ)
            @params power the power pin name (connected to Vdd - this library does power handling for the AR1020)
        */
        AR1020(PinName mosi, PinName miso, PinName clk, PinName enable, PinName siq, PinName power);
        ~AR1020();
        
        /**
            initialize the controller
        */
        virtual void init();
        
        /**
            read X coordinate (from last update)
            @returns last known X coordinate
        */
        virtual int x();
        /**
            read Y coordinate (from last update)
            @returns last known Y coordinate
        */
        virtual int y();
        
        /**
            @return 0 if pen is up, 1 if pen is down
        */
        virtual int pen();
        
        /**
            read coordinates on request
        */
        virtual void read();
        
        /**
            execute touch screen calibration
        */
        void calibrate();
    private:
        int cmd(char cmd,char* data, int len, bool doEnable=true);
        int readCalibResponse();
        int readByte();
        void writeByte(char bytee);

        SPI *_spi;
        DigitalOut *_enable;   
        InterruptIn *_irq;
        DigitalIn *_siq;
        int _x, _y, _pen; 
        
        DigitalOut *_led;
        DigitalOut* _power;
    
        DigitalOut *_mosi;
        DigitalIn *_miso;
        DigitalOut *_clk;
        
        bool _oldPen;
        TouchEvent _event;
};



#endif
