/*
* TimeZone library
* Copyright (c) 2010 Hendrik Lipka
* 
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
* 
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*/

#ifndef __TIME_H_
#define __TIME_H_

#include <time.h>
#include "stdio.h"
#include "string.h"
#include <list>

class Time;

/**
    This class encapsulates a point in time - which means it will not change after it has been created.
*/
class TimeStamp {
    friend class Time;
public:
    /**
        @returns the year of the time stamp
    */
    int getYear() {
        return _year+1900;
    };

    /**
        @returns the month of the time stamp (January is 1)
    */
    int getMonth() {
        return _mon+1;
    };

    int getDay() {
    /**
        @returns the day-of-the-month of the time stamp (starting with 1)
    */
        return _mday;
    };

    /**
        @returns the hour of the time stamp (0-23)
    */
    int getHour() {
        return _hour;
    };

    /**
        @returns the minute of the time stamp (0-59)
    */
    int getMinute() {
        return _min;
    };

    /**
        @returns the second of the time stamp (0-59)
    */
    int getSecond() {
        return _sec;
    };

    /**
        get the day of the week - monday is 0
        @returns the day-of-the-week of the time stamp
    */
    int getDayOfWeek() {
        int dow=_wday;
        dow--;
        if (dow==-1)
            dow=6;
        return dow;
    };

    /**
        @returns the day-of-the-year of the time stamp
    */
    /*
    int getDayOfYear() {
        return _yday;
    };
*/
    /**
        creates a new time stamp based on the UNIX time stamp (seconds since 1970)
    */
    TimeStamp(time_t unixTime) {
        updateTime(unixTime);
    };

    TimeStamp(int year, int mon, int day, int hour, int min, int sec, int wday)
    {
        _year=(char)(year-1900);
        _mon=(char)(mon-1);
        _mday=(char)day;
        _hour=(char)hour;
        _min=(char)min;
        _sec=(char)sec;
        _wday=(char)wday;
    }
    ~TimeStamp() {
    };
    
    /**
        @param ts the time stamp to compare to
        @returns true when both timestamp are the same point in time
    */
    bool isSame(TimeStamp* ts);
    
    /**
        @param ts the time stamp to compare to
        @return true when the current time stamp is before the given time stamp
    */
    bool isBefore(TimeStamp* ts);
    
    /**
        @returns time stamp transformed to ASCII (done by asctime() - but transformed to local time)
    */ 
    char *asChar() { struct tm * time=getTm();char *s=asctime(time); delete time;return s;}; 
    
    /**
        @param ts the time stamp to compare to
        @return true when the current time stamp is after the given time stamp
    */
    bool isAfter(TimeStamp* ts);
    int getStartOfDay()
    {
        tm* t=getTm();
        t->tm_hour=0;
        t->tm_min=0;
        t->tm_sec=0;
        return mktime(t);
    }
private:
    tm *getTm()
    {
        tm *time=new tm();
        
        time->tm_year=_year;
        time->tm_mon=_mon;
        time->tm_mday=_mday;
        time->tm_hour=_hour;
        time->tm_min=_min;
        time->tm_sec=_sec;
        time->tm_wday=_wday;
        
        return time;
    }
    void updateTime(time_t unixTime) {
        struct tm *time;
        time=localtime(&unixTime);
        _year=time->tm_year;
        _mon=time->tm_mon;
        _mday=time->tm_mday;
        _hour=time->tm_hour;
        _min=time->tm_min;
        _sec=time->tm_sec;
        _wday=time->tm_wday;
    }
    void print()
    {
        printf("ts=%i.%i.%i %i:%i.%i\n",_mday,_mon+1,_year+1900,_hour,_min,_sec);
    }
    // make sure this structure needs only 7 bytes - otherwise it's 48 bytes...
    char __packed _year;
    char __packed _mon;
    char __packed _mday;
    char __packed _hour;
    char __packed _min;
    char __packed _sec;
    char __packed _wday;
};

class TimeZoneEntry;

/**
    This class handles the time zone calculation, and is used for creating time stamp objects.
*/
class Time {
public:
    /**
        creates a new Time instance. On the first call, it reads the file 'timezone.csv' from the USB disk (local file system).
    */
    Time();
    ~Time();
    /**
        creates a new TimeStamp instance. The caller is responsible for deleting it afterwards!
        @returns the time stamp
    */
    TimeStamp* getTime();
    /**
        @returns the current time stamp (UTC) in UNIX format, without time zone correction.
    */
    long getUnixTime();
private:
    static TimeZoneEntry* _timeZoneEntries;
    static void readTimeZones();
    static int getTimeOffset(TimeStamp* ts);
};

#endif