
/*
Copyright (c) 2010 Donatien Garnier (donatiengar [at] gmail [dot] com)
 
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

/** \file
NTP Client header file
*/

#ifndef NTP_CLIENT_H
#define NTP_CLIENT_H

#include "core/net.h"
#include "core/netservice.h"
#include "api/UDPSocket.h"
#include "mbed.h"

///NTP Client results
enum NTPResult
{
  NTP_OK, ///<Success
  NTP_PROCESSING, ///<Processing
  NTP_PRTCL, ///<Protocol error
  NTP_TIMEOUT, ///<Connection timeout
  NTP_DNS ///<Could not resolve DNS hostname
};

  __packed struct NTPPacket //See RFC 4330 for Simple NTP
  {
    //WARN: We are in LE! Network is BE!
    //LSb first
    unsigned mode : 3;
    unsigned vn : 3;
    unsigned li : 2;
    
    uint8_t stratum;
    uint8_t poll;
    uint8_t precision;
    //32 bits header
    
    uint32_t rootDelay;
    uint32_t rootDispersion;
    uint32_t refId;
    
    uint32_t refTm_s;
    uint32_t refTm_f;
    uint32_t origTm_s;
    uint32_t origTm_f;
    uint32_t rxTm_s;
    uint32_t rxTm_f;
    uint32_t txTm_s;
    uint32_t txTm_f;
  };

/**
The NTP client is a simple UDP client that will update the mbed's RTC
*/
class NTPClient
{
public:
  /**
  Instantiates the NTP client
  */
  NTPClient();
  ~NTPClient();
  
  /**
      Updates the time using the server host, blocks until completion
      @param host : NTP server
  */
  NTPResult setTime(const Host& host); //Blocking
 
  void close();
  
private:
  void init();
  void open();
  
  /**
      Updates the time using the server host
      The function returns immediately and calls the previously set callback on completion or error
      @param host : NTP server
  */
  void doSetTime(const Host& host);

  void process(); //Main state-machine

  void setTimeout(int ms);
  void resetTimeout();
  
  void onTimeout(); //Connection has timed out
  void onUDPSocketEvent(UDPSocketEvent e);
  void onResult(NTPResult r); //Called when exchange completed or on failure
  
  NTPResult blockingProcess(); //Called in blocking mode, calls Net::poll() until return code is available

  UDPSocket* _pUDPSocket;
  
  

  enum NTPStep
  {
    NTP_PING,
    NTP_PONG
  };
  
  NTPStep _state;
  
  NTPPacket _pkt;
  
  Timer *_watchdog;
  int _timeout;
  
  bool _closed;
  
  Host _host;
  
  NTPResult _blockingResult; //Result if blocking mode

};

#endif
