#pragma diag_remark 1293
/*
Copyright (c) 2010 Donatien Garnier (donatiengar [at] gmail [dot] com)
 
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "NTPClient.h"
#include "dnsresolve.h"

#include <stdio.h>

//#define __DEBUG
#include "dbg/dbg.h"

#define NTP_PORT 123
#define NTP_CLIENT_PORT 0//50420 //Random port
#define NTP_REQUEST_TIMEOUT 5000
#define NTP_TIMESTAMP_DELTA 2208988800ull //Diff btw a UNIX timestamp (Starting Jan, 1st 1970) and a NTP timestamp (Starting Jan, 1st 1900)

#define htons( x ) ( (( x << 8 ) & 0xFF00) | (( x >> 8 ) & 0x00FF) )
#define ntohs( x ) (htons(x))

#define htonl( x ) ( (( x << 24 ) & 0xFF000000)  \
                   | (( x <<  8 ) & 0x00FF0000)  \
                   | (( x >>  8 ) & 0x0000FF00)  \
                   | (( x >> 24 ) & 0x000000FF)  )
#define ntohl( x ) (htonl(x))

NTPClient::NTPClient() : 
    _state(NTP_PING), 
    _timeout(NTP_REQUEST_TIMEOUT), 
    _closed(true), 
    _host(), 
    _blockingResult(NTP_PROCESSING)
{
  _watchdog=new Timer();
  DBG("\r\nNew NTPClient %p\r\n",this);
}

NTPClient::~NTPClient()
{
  close();
  delete _watchdog;
}

//High level setup functions
NTPResult NTPClient::setTime(const Host& host) //Blocking
{
  doSetTime(host);
  return blockingProcess();
}

void NTPClient::doSetTime(const Host& host)
{
  init();
  resetTimeout();
  _host = host;
  if(!_host.getPort())
  {
    _host.setPort(NTP_PORT);
  }
  if(_host.getIp().isNull())
  {
  
    DNSResolver *dr=new DNSResolver();
    IpAddr ad=dr->resolveName(host.getName());
    delete dr;
    if (ad.isNull())
    {
        onResult(NTP_DNS);
        return;
    }
    _host.setIp(ad);
  }
  open();
}

void NTPClient::close()
{
  if(_closed)
    return;
  _closed = true; //Prevent recursive calling or calling on an object being destructed by someone else
  _watchdog->stop();
  _pUDPSocket->resetOnEvent();
  _pUDPSocket->close();
  delete _pUDPSocket;
}

void NTPClient::init() //Create and setup socket if needed
{
  if(!_closed) //Already opened
    return;
  _state = NTP_PING;
  _pUDPSocket = new UDPSocket;
  _pUDPSocket->setOnEvent(this, &NTPClient::onUDPSocketEvent);
  _closed = false;
  DBG("NTPClient: Init OK\n");
}

void NTPClient::open()
{
  resetTimeout();
  DBG("Opening connection\n");
  _state = NTP_PING;
  Host localhost(IpAddr(), NTP_CLIENT_PORT, "localhost"); //Any local address
  _pUDPSocket->bind(localhost);
  if ((int)time(NULL) < 1280000000) set_time( 1280000000 ); //End of July 2010... just there to limit offset range

  process();

}

#define MIN(a,b) ((a)<(b))?(a):(b)
void NTPClient::process() //Main state-machine
{
  int len;
  Host host;
  switch(_state)
  {
  case NTP_PING:
    DBG("Ping\r\n");
    //Prepare NTP Packet:
    _pkt.li = 0; //Leap Indicator : No warning
    _pkt.vn = 4; //Version Number : 4
    _pkt.mode = 3; //Client mode
    _pkt.stratum = 0; //Not relevant here
    _pkt.poll = 0; //Not significant as well
    _pkt.precision = 0; //Neither this one is
    
    _pkt.rootDelay = 0; //Or this one
    _pkt.rootDispersion = 0; //Or that one
    _pkt.refId = 0; //...
    
    _pkt.refTm_s = 0;
    _pkt.origTm_s = 0;
    _pkt.rxTm_s = 0;
    _pkt.txTm_s = htonl( NTP_TIMESTAMP_DELTA + time(NULL) ); //WARN: We are in LE format, network byte order is BE
    
    _pkt.refTm_f = _pkt.origTm_f = _pkt.rxTm_f = _pkt.txTm_f = 0;
    
    #ifdef __DEBUG
    //Hex Dump:
    DBG("Dump Tx:\r\n");
    for(int i = 0; i< sizeof(NTPPacket); i++)
    {
      DBGL("%02x ", *((char*)&_pkt + i));
    }
    DBGL("\r\n");
    #endif
    
    len = _pUDPSocket->sendto( (char*)&_pkt, sizeof(NTPPacket), &_host );
    if(len < sizeof(NTPPacket))
      { onResult(NTP_PRTCL); close(); return; }
      
    _state = NTP_PONG; 
          
    break;
  
  case NTP_PONG:
    DBG("Pong\r\n");
    while( len = _pUDPSocket->recvfrom( (char*)&_pkt, sizeof(NTPPacket), &host ) )
    {
      if( len <= 0 )
        break;
      if( !host.getIp().isEq(_host.getIp()) )
        continue; //Not our packet
      if( len > 0 )
        break;
    }
    
    if(len == 0)
      return; //Wait for the next packet
    
    if(len < 0)
      { onResult(NTP_PRTCL); close(); return; }
    
    if(len < sizeof(NTPPacket)) //TODO: Accept chunks
      { onResult(NTP_PRTCL); close(); return; }
      
    #ifdef __DEBUG
    //Hex Dump:
    DBG("Dump Rx:\r\n");
    for(int i = 0; i< sizeof(NTPPacket); i++)
    {
      DBGL("%02x ", *((char*)&_pkt + i));
    }
    DBGL("\r\n");
    #endif
      
    if( _pkt.stratum == 0)  //Kiss of death message : Not good !
    {
      onResult(NTP_PRTCL); close(); return;
    }
    
    //Correct Endianness
    _pkt.refTm_s = ntohl( _pkt.refTm_s );
    _pkt.refTm_f = ntohl( _pkt.refTm_f );
    _pkt.origTm_s = ntohl( _pkt.origTm_s );
    _pkt.origTm_f = ntohl( _pkt.origTm_f );
    _pkt.rxTm_s = ntohl( _pkt.rxTm_s );
    _pkt.rxTm_f = ntohl( _pkt.rxTm_f );
    _pkt.txTm_s = ntohl( _pkt.txTm_s );
    _pkt.txTm_f = ntohl( _pkt.txTm_f );
    
    //Compute offset, see RFC 4330 p.13
    uint32_t destTm_s = (NTP_TIMESTAMP_DELTA + time(NULL));
    //int32_t origTm = (int32_t) ((uint64_t) _pkt.origTm - NTP_TIMESTAMP_DELTA); //Convert in local 32 bits timestamps
    //int32_t rxTm = (int32_t) ((uint64_t) _pkt.rxTm - NTP_TIMESTAMP_DELTA); //Convert in local 32 bits timestamps
    //int32_t txTm = (int32_t) ((uint64_t) _pkt.txTm - NTP_TIMESTAMP_DELTA); //Convert in local 32 bits timestamps
   // int64_t offset = ( ( ( _pkt.rxT_s - m_pkt.origTm_s ) + ( m_pkt.txT_s - destTm_s ) ) << 32 + ( ( m_pkt.rxTm_f - m_pkt.origTm_f ) + ( m_pkt.txT_f - 0 ) ) ) / 2;
    int64_t offset = ( (int64_t)( _pkt.rxTm_s - _pkt.origTm_s ) + (int64_t) ( _pkt.txTm_s - destTm_s ) ) / 2; //Avoid overflow
    DBG("Sent @%d\r\n", _pkt.txTm_s);
    DBG("Offset: %d\r\n", offset);

    //Set time accordingly
    set_time( time(NULL) + (offset /*>> 32*/) );
    
    onResult(NTP_OK);
    close();
    break;
  }
}

void NTPClient::setTimeout(int ms)
{
  _timeout = ms;
}

void NTPClient::resetTimeout()
{
  _watchdog->reset();
  _watchdog->start();
}

void NTPClient::onTimeout() //Connection has timed out
{
  close();
  onResult(NTP_TIMEOUT);
}

void NTPClient::onUDPSocketEvent(UDPSocketEvent e)
{
  resetTimeout();
  switch(e)
  {
  case UDPSOCKET_READABLE: //The only event for now
    process();
    break;
  }
}

void NTPClient::onResult(NTPResult r) //Must be called by impl when the request completes
{
  _blockingResult = r; //Blocking mode
}

NTPResult NTPClient::blockingProcess() //Called in blocking mode, calls Net::poll() until return code is available
{
  _blockingResult = NTP_PROCESSING;
  do
  {
    Net::poll();
    wait_us(100);
    if (_watchdog->read_ms()>_timeout)
        return NTP_TIMEOUT;
  } while(_blockingResult == NTP_PROCESSING);
 
  Net::poll(); //Necessary for cleanup

  return _blockingResult;
}
