/*
 * mbed LCDWindow library
* Copyright (c) 2010 Hendrik Lipka
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*/

#include "sed1335text.h"

#include "BusOut.h"
#include "DigitalOut.h"
#include "wait_api.h"

#define FIRSTBANK_MEM 0
#define SECONDBANK_MEM 0x1000
#define THIRDBANK_MEM 0x800

#define HIGH(x) ((x&0xff00)>>8)
#define LOW(x) (x&0xff)

void SED1335TextLCD::writeText(const unsigned int column, const unsigned int row, const char text[]) {
    int i=0;
    
    int pos=row*getColumns()+column;

    if (!_guard->take())
        return;
    
    sendCmd(0x46); // set cursor addr
    sendData(LOW(pos)); 
    sendData(HIGH(pos));
    
    sendCmd(0x42);
    while (text[i]!=0) {
        sendData(text[i]);
        i++;
    }
    _guard->release();
}

void SED1335TextLCD::clearBank(int start, int size, int data) {
    sendCmd(0x46); // set cursor addr
    sendData(LOW(start)); // start of RAM
    sendData(HIGH(start));
    wait_ms(1);
    
    sendCmd(0x42); // write to memory
    for (int i=0;i<size;i++)
        sendData(data);    // send space
    wait_ms(1);
}

void SED1335TextLCD::clear() {
    _guard->take();

    clearBank(FIRSTBANK_MEM,getColumns()*getRows(),0x20);    
    clearBank(SECONDBANK_MEM,getColumns()*getRows()*8,0);    
    clearBank(THIRDBANK_MEM,getColumns()*getRows(),0);    
    
    _guard->release();
}


void SED1335TextLCD::character(int column, int row, int c){
    _guard->take();
    _guard->release();
}

SED1335TextLCD::SED1335TextLCD
(const unsigned int columns, const unsigned int rows, BusOut *data, const PinName read, const PinName write, const PinName cs, const PinName a0, const PinName reset)
        :TextLCDBase(columns, rows) {
    _data=data;
    _rd=new DigitalOut(read);
    _wr=new DigitalOut(write);
    _a0=new DigitalOut(a0);
    _cs=new DigitalOut(cs);
    _reset=new DigitalOut(reset);
    wait_ms(100);
}

void SED1335TextLCD::init() {
    initInternal();
    initInternal();
}

void SED1335TextLCD::initInternal() {
    _reset->write(0);
    wait_ms(2);
    _reset->write(1);
    wait_ms(10);
    
    sendCmd(0x40); // SYSTEM SET
    sendData(0x30); // p1 set cg rom
    sendData(0x87); // p2 
    sendData(0x7);  // p3 
    sendData(getColumns()-1); // p4 
    sendData(getColumns()+3); // p5 
    sendData(0xef); // p6 
    sendData(getColumns()); // p7 
    sendData(0x0);  // p8 
    wait_ms(1);

    sendCmd(0x44); // SCROLL
    sendData(LOW(FIRSTBANK_MEM)); // first block
    sendData(HIGH(FIRSTBANK_MEM));
    sendData(getRows()*8); // lines
    sendData(LOW(SECONDBANK_MEM)); // second block
    sendData(HIGH(SECONDBANK_MEM));
    sendData(getRows()*8); // lines
    sendData(LOW(THIRDBANK_MEM)); // third block at 0x800
    sendData(HIGH(THIRDBANK_MEM));
    wait_ms(1);
    
    sendCmd(0x5a); // HDOT SCR
    sendData(0x0);
        

    sendCmd(0x5b); // OVLAY
    sendData(0x01); // 2 layers, text, x-or
    wait_ms(1);

    sendCmd(0x59); // DISP on
    sendData(0x54); // all layers on, cursor off
//    sendData(0x56); // all layers on, cursor flashing
    wait_ms(1);
    
    clear();
    
    sendCmd(0x5d); // CSR FORM
    sendData(0x04); // 8 pixels wide
    sendData(0x86); // 2 pixel high, block cursor
    wait_ms(1);
    
    sendCmd(0x4c); // CSR DIR right
    wait_ms(1);
    
}

void SED1335TextLCD::sendCmd(const unsigned char cmd) {
    _rd->write(1);
    _wr->write(1);
    _a0->write(1);
    _cs->write(0);
    // address setup time is 0
    sendByte(cmd);
    wait_us(10);
}

void SED1335TextLCD::sendData(const unsigned char cmd) {
    _rd->write(1);
    _wr->write(1);
    _a0->write(0);
    _cs->write(0);
    // address setup time is 0
    sendByte(cmd);
    wait_us(10);
}

void SED1335TextLCD::sendByte(const unsigned char byte) {
    // display reads with rising flank of /wr
    // first, set data, then set /wr low (no timing needed)
    _data->write(byte);
    _wr->write(0);
    // wait for setup
    wait_us(1);
    // set /wr back to high
    _wr->write(1);
    // address / data hold time is 10 / 5 ns, so its handled by executing normal code :)
}
