/* mbed RC6 (Philips remote control protocol) library
 * Copyright (c) 2011 Jeroen Hilgers
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
 #ifndef __RC6_H__
 #define __RC6_H__
 
 #include "mbed.h"

/** RC6 (Philips remote control) receiver / transmitter library.
  *
  * Example: Code receiver for 'play/pause' on remote for Philips HDD&DVD recorder.
  * 
  * Leader  |S3|S2|S1|S0|Togl|A7|A6|A5|A4|A3|A2|A1|A0|D7|D6|D5|D4|D3|D2|D1|D0|
  *         |1 |0 |0 |0 |1   |0 |0 |1 |1 |0 |0 |1 |0 |0 |0 |1 |0 |1 |1 |0 |0 |
  *         |  |  |  |  |    |  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |
  * ------  |- | -| -| -|--  | -| -|- |- | -| -|- | -| -| -|- | -|- |- | -| -|
  *       --| -|- |- |- |  --|- |- | -| -|- |- | -|- |- |- | -|- | -| -|- |- |
  *         |  |  |  |  |    |  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |
  *
  * The toggle bit is in bit 16, A7-A0 in bits 15-8, D7-D0 in bits 7-0.
  * Thus for the example, 0x1322C is returned. S3..S0 are not implemented
  * and received / transmitted as shown in the example.
  */
  
 class Rc6Transmitter
 {
    public:
        /** Create a Rc6Transmitter and initizalize it.
         *
         * @param pin irPin pin connected IR Led (led on on high level).
         */
        Rc6Transmitter(PinName irPin);

        /** Send a code once. 
         * The function will return synchronously and send the code once. 
         *
         * @param code Code to send.
         */
        void Send(uint32_t code);
        
        /** Check if the transmitter is idle.
         * 
         * @return True if transmitter still busy.
         */
        bool IsBusy();
    
    private:
        void Tick();
    
        bool mBusy;
        uint16_t mPattern[50];  // 1 leader, 4 startbits, 1 toggle bit, 16 data bits ==>
                                // at most 24 mark / pause durations + 0,0 terminator = 50 entries. 
        Ticker mTicker;
        uint16_t mMark;          // Tick: Remaining half-cycles of mark.
        uint16_t mPause;         // Tick: Remaining half-cycles of pause.        
        uint16_t *mPatPtr;       // Transitter pointer.
        DigitalOut mOut;     // Pin connected to IR.
};
 
 class Rc6Receiver
 {
    public:
        /** Create a Rc6Receiver and initizalize it.
         *
         * @param pin irPin pin connected to IR sensor (low level when IR is received, e.g. TSOP1736).
         */
        Rc6Receiver(PinName irPin);

        /** Check if anything was received since last poll. Returns 0xFFFFFFFF if not.
         *
         * @return Last code received or 0xFFFFFFFF if nothing was received since last poll.
         */
        uint32_t Receive();
        
    private:
        InterruptIn irq;
        void OnRise();
        void OnFall();
        void OnTimeout();
        Timeout mTimeout;
        Timer mTimer;
        bool mBusy;
        uint32_t mCode;  
        uint32_t mLastCode;
 };
  
 #endif // __RC6_H__