/* mbed Samsung LCD Library, for a samsung KS0723 based 128x64 black/white lcd panel.
 * Copyright (c) 2011, Jeroen Hilgers
 *
 * I based the interface on the LCD libraries by Simon Ford, to obtain compatibility
 * between different displays. I used TextLCD as an example to implement the 
 * : public Steam part of this library.
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "LcdKs0723.h"

LcdKs0723::LcdKs0723(PinName reset, PinName rs, PinName write, PinName read, PinName d0, PinName d1,
                       PinName d2, PinName d3, PinName d4, PinName d5, PinName d6, PinName d7) :
            mReset(reset), mRs(rs), mWrite(write), mRead(read),
            mData(d0, d1, d2, d3, d4, d5, d6, d7)
{
    mRow = 0;
    mCol = 0;

    // Init IO
    mWrite = 1;
    mRead = 1;
    mRs = 0;
    mData.input();
    
    // Reset
    mReset = 0;
    wait(0.001);
    mReset = 1;
    wait(0.1);

    // Select LCD BIAS
    WriteControl(0xA3); 

    // ADCSelect
    WriteControl(0xA1);

    // PowerControl
    WriteControl(0x2C);
    wait(0.002);
    WriteControl(0x2E);
    wait(0.002);
    WriteControl(0x2F);
    wait(0.002);

    // Regulator resistor
    WriteControl(0x24);

    // Reference Voltage
    WriteControl(0x81);
    WriteControl(0x20);
    wait(0.1);

    cls();

    // Display on.
    WriteControl(0xAF); 
} 

int LcdKs0723::_putc(int value)
{
  if(value == '\n') 
  {
    mCol = 0;
    mRow = (mRow+1) & 7;
    return value;
  }

  const uint8_t *ptr = mFont5x7 + value*5;
  uint8_t x;
  
  WriteControl( 0xB0 | mRow); // Page address.
  WriteControl( 0x10 | (((mCol+3)>>4)&0x0F) );        // Column address. Display starts at pixel 3.
  WriteControl( 0x00 | ((mCol+3)&0x0F) );           // Column address 2nd nibble. 

  mRs = 1;
  mData.output();

  for(x=0; x<5; x++)
  {
     mData = *ptr++;
     mWrite = 0;
     mWrite = 1;
  }
  mData = 0;
  mWrite = 0;
  mWrite = 1;

  mData.input();
  mData = 0;
  mCol += 6;
  
  if(mCol > 122) 
  {
    mCol = 0;
    mRow = (mRow+1)&7;
  } 
  
  return value;
}

void LcdKs0723::locate(int column, int row)
{
    mRow = row & 7; // Limit to 0-7.
    if(column < 21) 
        mCol = column*6;
    else
        mCol = 0;
}

void LcdKs0723::cls()
{
  uint16_t page;
  uint8_t x;
  for(page = 0; page<8; page++)
  {
    WriteControl(0xB0 | page); // Page address.
    WriteControl(0x10);        // Column address. Start at column 0.
    WriteControl(0);           // Column address 2nd nibble. 

    // Setup to write data 0x00 to display.
    mRs = 1;
    mData.output();
    mData = 0;

    // First three columns are attached to '>' symbols next to the pixel area.
    // This library has no use for them, but we don't want them uninitialized.
    for(x=0; x<3; x++)
    {
      mWrite = 0;
      mWrite = 1;
    }
    
    // Columns 3-131 display pixel data. Clear them.
    for(x=0; x<128; x++)
    {
        mWrite = 0;
        mWrite = 1;
    }
    
    // Last five three columns are attached to '<' symbols next to the pixel area.
    // This library has no use for them, but we don't want them uninitialized.
    for(x=0; x<5; x++)
    {
      mWrite = 0;
      mWrite = 1;
    }
    mData.input();
    mData = 0;
  }

  mRow =0;
  mCol = 0;
}

uint8_t LcdKs0723::ReadData (void)
{
  uint8_t result;
  mData.input();
  mRs = 1;
  mRead = 0;
  result = mData;
  mRead = 1;
  return result;
}

void LcdKs0723::WriteData (uint8_t value)
{
  mRs = 1;
  mData.output();
  mData = value;
  mWrite = 0;
  mWrite = 1;
  mData.input();
  mData = 0;
}

uint8_t LcdKs0723::ReadStatus(void)
{
  uint8_t result;
  mData.input();
  mRs = 0;
  mRead = 0;
  result = mData;
  mRead = 1;
  return result;
}

void LcdKs0723::WriteControl(uint8_t value)
{
  mRs = 0;
  mData.output();
  mData = value;
  mWrite = 0;
  mWrite = 1;
  mData.input();
  mData = 0;
}

const uint8_t LcdKs0723::mFont5x7[] = 
{
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x07, 0x05, 0x07, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00, // 1A = deg symbol.
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x4F, 0x00, 0x00,
    0x00, 0x07, 0x00, 0x07, 0x00,    0x14, 0x7F, 0x14, 0x7F, 0x14,
    0x24, 0x2A, 0x7F, 0x2A, 0x12,    0x23, 0x13, 0x08, 0x64, 0x62,
    0x36, 0x49, 0x55, 0x22, 0x50,    0x00, 0x05, 0x03, 0x00, 0x00,
    0x00, 0x1C, 0x22, 0x41, 0x00,    0x00, 0x41, 0x22, 0x1C, 0x00,
    0x14, 0x08, 0x3E, 0x08, 0x14,    0x08, 0x08, 0x3E, 0x08, 0x08,
    0x00, 0x50, 0x30, 0x00, 0x00,    0x08, 0x08, 0x08, 0x08, 0x08,
    0x00, 0x60, 0x60, 0x00, 0x00,    0x20, 0x10, 0x08, 0x04, 0x02,
    0x3E, 0x51, 0x49, 0x45, 0x3E,    0x00, 0x42, 0x7F, 0x40, 0x00,
    0x42, 0x61, 0x51, 0x49, 0x46,    0x21, 0x41, 0x45, 0x4B, 0x31,
    0x18, 0x14, 0x12, 0x7F, 0x10,    0x27, 0x45, 0x45, 0x45, 0x39,
    0x3C, 0x4A, 0x49, 0x49, 0x30,    0x01, 0x71, 0x09, 0x05, 0x03,
    0x36, 0x49, 0x49, 0x49, 0x36,    0x06, 0x49, 0x49, 0x29, 0x1E,
    0x00, 0x36, 0x36, 0x00, 0x00,    0x00, 0x56, 0x36, 0x00, 0x00,
    0x08, 0x14, 0x22, 0x41, 0x00,    0x14, 0x14, 0x14, 0x14, 0x14,
    0x00, 0x41, 0x22, 0x14, 0x08,    0x02, 0x01, 0x51, 0x09, 0x06,
    0x32, 0x49, 0x79, 0x41, 0x3E,    0x7E, 0x11, 0x11, 0x11, 0x7E,
    0x7F, 0x49, 0x49, 0x49, 0x36,    0x3E, 0x41, 0x41, 0x41, 0x22,
    0x7F, 0x41, 0x41, 0x22, 0x1C,    0x7F, 0x49, 0x49, 0x49, 0x41,
    0x7F, 0x09, 0x09, 0x09, 0x01,    0x3E, 0x41, 0x49, 0x49, 0x7A,
    0x7F, 0x08, 0x08, 0x08, 0x7F,    0x00, 0x41, 0x7F, 0x41, 0x00,
    0x20, 0x40, 0x41, 0x3F, 0x01,    0x7F, 0x08, 0x14, 0x22, 0x41,
    0x7F, 0x40, 0x40, 0x40, 0x40,    0x7F, 0x02, 0x0C, 0x02, 0x7F,
    0x7F, 0x04, 0x08, 0x10, 0x7F,    0x3E, 0x41, 0x41, 0x41, 0x3E,
    0x7F, 0x09, 0x09, 0x09, 0x06,    0x3E, 0x41, 0x51, 0x21, 0x5E,
    0x7F, 0x09, 0x19, 0x29, 0x46,    0x46, 0x49, 0x49, 0x49, 0x31,
    0x01, 0x01, 0x7F, 0x01, 0x01,    0x3F, 0x40, 0x40, 0x40, 0x3F,
    0x1F, 0x20, 0x40, 0x20, 0x1F,    0x3F, 0x40, 0x38, 0x40, 0x3F,
    0x63, 0x14, 0x08, 0x14, 0x63,    0x07, 0x08, 0x70, 0x08, 0x07,
    0x61, 0x51, 0x49, 0x45, 0x43,    0x00, 0x7F, 0x41, 0x41, 0x00,
    0x02, 0x04, 0x08, 0x10, 0x20,    0x00, 0x41, 0x41, 0x7F, 0x00,
    0x04, 0x02, 0x01, 0x02, 0x04,    0x40, 0x40, 0x40, 0x40, 0x40,
    0x00, 0x01, 0x02, 0x04, 0x00,    0x20, 0x54, 0x54, 0x54, 0x78,
    0x7F, 0x48, 0x44, 0x44, 0x38,    0x38, 0x44, 0x44, 0x44, 0x20,
    0x38, 0x44, 0x44, 0x48, 0x7F,    0x38, 0x54, 0x54, 0x54, 0x18,
    0x08, 0x7E, 0x09, 0x01, 0x02,    0x0C, 0x52, 0x52, 0x52, 0x3E,
    0x7F, 0x08, 0x04, 0x04, 0x78,    0x00, 0x44, 0x7D, 0x40, 0x00,
    0x20, 0x40, 0x44, 0x3D, 0x00,    0x7F, 0x10, 0x28, 0x44, 0x00,
    0x00, 0x41, 0x7F, 0x40, 0x00,    0x7C, 0x04, 0x18, 0x04, 0x78,
    0x7C, 0x08, 0x04, 0x04, 0x78,    0x38, 0x44, 0x44, 0x44, 0x38,
    0x7C, 0x14, 0x14, 0x14, 0x08,    0x08, 0x14, 0x14, 0x18, 0x7C,
    0x7C, 0x08, 0x04, 0x04, 0x08,    0x48, 0x54, 0x54, 0x54, 0x20,
    0x04, 0x3F, 0x44, 0x40, 0x20,    0x3C, 0x40, 0x40, 0x20, 0x7C,
    0x1C, 0x20, 0x40, 0x20, 0x1C,    0x3C, 0x40, 0x30, 0x40, 0x3C,
    0x44, 0x28, 0x10, 0x28, 0x44,    0x0C, 0x50, 0x50, 0x50, 0x3C,
    0x44, 0x64, 0x54, 0x4C, 0x44,    0x00, 0x08, 0x36, 0x41, 0x00,
    0x00, 0x00, 0x7F, 0x00, 0x00,    0x00, 0x41, 0x36, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00,    0x00, 0x00, 0x00, 0x00, 0x00
};

 
    