/* mbed Samsung LCD Library, for a samsung KS0723 based 128x64 black/white lcd panel.
 * Copyright (c) 2011, Jeroen Hilgers
 *
 * I based the interface on the LCD libraries by Simon Ford, to obtain compatibility
 * between different displays. I used TextLCD as an example to implement the 
 * : public Steam part of this library.
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


#ifndef MBED_SAMSUNG_LCD_H
#define MBED_SAMSUNG_LCD_H

#include "mbed.h"

/** LcdKs0723 class. Provides an interface compatible with
 ** other display libraries by Simon Ford on www.mbed.org.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "LcdKs0723.h"
 *
 * LcdKs0723 lcd(p28, p10, p27, p19, p25, p15, p24, p16, p23, p17, p22, p18);
 *
 * int main() 
 * {
 *    lcd.printf("KS0723 LCD Display driver.\n");
 *    lcd.printf("Float: %1.3f\n", 1.375f);
 *    lcd.printf("Hex: 0x%04X\n", 0xB12);
 *    while(1) 
 *   {
 *   }
 * }
 * @endcode
 */

class LcdKs0723 : public Stream 
{
  public:
    /** Create a LcdKs0723 interface
     *
     * @param reset Reset line
     * @param rs    RS line
     * @param write Write line
     * @param read  Read line
     * @param d0-d7 Data lines
     */  
    LcdKs0723(PinName reset, PinName rs, PinName write, PinName read, PinName d0, PinName d1,
               PinName d2, PinName d3, PinName d4, PinName d5, PinName d6, PinName d7);
               
#if DOXYGEN_ONLY
    /** Write a character to the LCD
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formated string to the LCD
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formating the string.
     */
    int printf(const char* format, ...);
#endif               

    /** Locate to a screen column and row
     *
     * @param column  The horizontal position from the left, indexed from 0
     * @param row     The vertical position from the top, indexed from 0
     */
    void locate(int column, int row);

    /** Clear the screen and locate to 0,0 */
    void cls();

  private:
     DigitalOut mReset, mRs, mWrite, mRead;
     BusInOut mData;
     int mRow;
     int mCol;
     
     uint8_t ReadData (void);
     void WriteData (uint8_t value);
     uint8_t ReadStatus(void);
     void WriteControl(uint8_t value);
     
     // Character definition.
     static const uint8_t mFont5x7[];
     
  protected:
     // Stream implementation functions.
     virtual int _putc(int value);
     virtual int _getc() {return -1;}     
};

#endif
