//
// MessagePack for C++ static resolution routine
//
// Copyright (C) 2014-2015 KONDO Takatoshi
//
//    Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//    http://www.boost.org/LICENSE_1_0.txt)
//
#ifndef MSGPACK_TYPE_VECTOR_UNSIGNED_CHAR_HPP
#define MSGPACK_TYPE_VECTOR_UNSIGNED_CHAR_HPP

#include "msgpack/versioning.hpp"
#include "msgpack/adaptor/adaptor_base.hpp"
#include "msgpack/adaptor/check_container_size.hpp"

#include <vector>

namespace msgpack {

/// @cond
MSGPACK_API_VERSION_NAMESPACE(v1) {
/// @endcond

namespace adaptor {

template <typename Alloc>
struct convert<std::vector<unsigned char, Alloc> > {
    msgpack::object const& operator()(msgpack::object const& o, std::vector<unsigned char, Alloc>& v) const {
        switch (o.type) {
        case msgpack::type::BIN:
            v.resize(o.via.bin.size);
            std::memcpy(&v.front(), o.via.bin.ptr, o.via.bin.size);
            break;
        case msgpack::type::STR:
            v.resize(o.via.str.size);
            std::memcpy(&v.front(), o.via.str.ptr, o.via.str.size);
            break;
        default:
            printf("throw msgpack::type_error()");
//            throw msgpack::type_error();
            break;
        }
        return o;
    }
};

template <typename Alloc>
struct pack<std::vector<unsigned char, Alloc> > {
    template <typename Stream>
    msgpack::packer<Stream>& operator()(msgpack::packer<Stream>& o, const std::vector<unsigned char, Alloc>& v) const {
        uint32_t size = checked_get_container_size(v.size());
        o.pack_bin(size);
        o.pack_bin_body(reinterpret_cast<char const*>(&v.front()), size);

        return o;
    }
};

template <typename Alloc>
struct object<std::vector<unsigned char, Alloc> > {
    void operator()(msgpack::object& o, const std::vector<unsigned char, Alloc>& v) const {
        uint32_t size = checked_get_container_size(v.size());
        o.type = msgpack::type::BIN;
        o.via.bin.ptr = reinterpret_cast<char const*>(&v.front());
        o.via.bin.size = size;
    }
};

template <typename Alloc>
struct object_with_zone<std::vector<unsigned char, Alloc> > {
    void operator()(msgpack::object::with_zone& o, const std::vector<unsigned char, Alloc>& v) const {
        uint32_t size = checked_get_container_size(v.size());
        o.type = msgpack::type::BIN;
        char* ptr = static_cast<char*>(o.zone.allocate_align(size));
        o.via.bin.ptr = ptr;
        o.via.bin.size = size;
        std::memcpy(ptr, &v.front(), size);
    }
};

} // namespace adaptor

/// @cond
} // MSGPACK_API_VERSION_NAMESPACE(v1)
/// @endcond

} // namespace msgpack

#endif // MSGPACK_TYPE_VECTOR_UNSIGNED_CHAR_HPP
