 /* mbed Library - Capture driver for CT32B0
 * Copyright (c) 2010
 * released under MIT license http://mbed.org/licence/mit
 */
 /***********************************************************************//**
 * @file        capturet1_11u24.cpp
 * @brief       Driver file for the CAP1 function of CT32B0 in the  mbed 11U24. 
 * @version     0.0
 * @date        05 Feb 2014
 * @author      
 **************************************************************************/
#include "mbed.h"
#include "capturet0_11u24.h"

Capture *Capture::instance;

/*********************************************************************//**
 * @brief       Create a Capture object and configure it. 
 * @param       None   Uses p7 as the Capture input
 * @return      None
 **********************************************************************/
Capture::Capture(void)
{       
    /* Set up clock and power, using default peripheral system clock divider */   
    LPC_SYSCON->SYSAHBCLKCTRL |=   CT32B0_CLK_ENABLE; // See page 30 of the 11U24 User Guide
    // Check the clock divider ratio by reading   LPC_SYSCON->SYSAHBCLKDIV ls byte
    LPC_CT32B0->PR  =  0;                           // Prescale counter divide by 1
       
    /* Configure IOCON for the proper function of the capture pin. Use a pull-up.     */
    LPC_IOCON->PIO1_29 = (CT32B0_CAP1 | CAP_PULLUP); // See page 114 of the 11U24 User Guide   
              
    // Select the capture edge and int mode
    LPC_CT32B0->CCR =   CCR_CAP1FE  | CCR_CAP1I;    // Select inputs and interrupts. See page 345
    
    // Set up the rising edge clear, using timer mode
    LPC_CT32B0->CTCR = CTCR_CTM_TIMER | CTCR_ENCC | CTCR_SEICC_CAP1RE; // See page 350
    
        
    // Clear interrupt flag
    LPC_CT32B0->IR = IR_CR1INT;     // See page 342             
  
    /* preemption = 1, sub-priority = 1 */
//    NVIC_SetPriority(TIMER2_IRQn, ((0x01<<3)|0x01));
//     NVIC_SetPriority(TIMER2_IRQn, 0);          // Set to top priority to minimize interference from tickers.
     
    //* Attach IRQ
    instance = this;
    NVIC_SetVector(TIMER_32_0_IRQn, (uint32_t)&_Captureisr);                   
  
}

/*********************************************************************//**
 * @brief        Start capturing data. Average the specified number of samples. 
 * @param        (int) numsamples    Log base 2 of the number of samples to be averaged
 * @return        None
 **********************************************************************/
void Capture::Start(int numsamples) {
    Nsamples = numsamples;
    capturedata = 0;                        // Clear the accumulator
    flag = 0;                               // No data ready
    
    /* Enable interrupt for CAPTURE  */
    NVIC_EnableIRQ(TIMER_32_0_IRQn); 
    
    // Start the timer
    LPC_CT32B0->TCR =  TCR_CEN; // enable
}

/*********************************************************************//**
 * @brief        Read (accumulated) sample count. 
 * 
 * @return       (unsigned int) Accumulated capture value
 **********************************************************************/
unsigned int Capture::Read(void) {
    flag = 0;                   // Clear previous status (or error)
    return capturedata;
}

/*********************************************************************//**
 * @brief        Check for accumulation complete (data is ready to read). 
 * 
 * @return       (unsigned int) Accumulated capture value
 **********************************************************************/
int Capture::Ready(void) {
    return flag;
}


/* Capture isr instantiator */    
void Capture::_Captureisr(void)
{
    instance->Captureisr();
}

/*********************************************************************//**
 * @brief        Capture interrupt service routine. Handles accumulation.
 * 
 * @param[in]    none
 *                                 
 * @return       none
 **********************************************************************/
void Capture::Captureisr(void) {
    static unsigned int samplecount = 0;   // Counts passes
    static unsigned int capturesum = 0;    // Accumulates the readings
    
    if(LPC_CT32B0->IR & IR_CR1INT) {
        LPC_CT32B0->IR |= IR_CR1INT;       // Clear the interrupt request
        capturesum += LPC_CT32B0->CR2;     // Read and accumulate the captured results
                                            // It is necessary to use "CR2" in order to get the proper
                                            //  offset from the base address (an issue with CMSIS)
        if(++samplecount == Nsamples) {
            capturedata = capturesum;      // Update the output data
            samplecount = 0;               // Restart the pass counter
            capturesum = 0;                // Re-initialize the running total
            if(flag != 0) flag = -1;       // Make flag negative on an overrun
            else flag = 1;                 // Set flag to 1 if no error
        }     
    }
    return;
}