/**
 * C1098-SS device driver class (Version 1.0)
 * Reference documents: C1098-SS User Manual v1.0 2012.5.6
 *
 * CameraC328Library
 * Copyright (C) 2010 Shinichiro Nakamura (CuBeatSystems)
 * http://shinta.main.jp/
 *
 * CameraC1098-SS Library
 * Copyright (C) 2012 Tadao Iida
 */

#include "mbed.h"
#include "SerialBuffered.h"

#ifndef _CAMERA_C1098_H_
#define _CAMERA_C1098_H_

/*
 * Class: CameraC1098
 */
 
class CameraC1098 {
public:

    /**
     * @enum JPEG resolution.
     */
    enum JpegResolution {
         JpegResolution80x64 = 0x01,   // unofficial
         JpegResolution160x128 = 0x03, // unofficial
         JpegResolution320x240 = 0x05, // QVGA
         JpegResolution640x480 = 0x07  // VGA
    };

    /**
     * @enum Error number.
     */
    enum ErrorNumber {
        NoError = 0x00,
        UnexpectedReply = 0x04,
        ParameterError = 0x0b,
        SendRegisterTimeout = 0x0c,
        CommandIdError = 0x0d,
        CommandHeaderError = 0xf0,
        SetTransferPackageSizeWrong = 0x11
    };

    /**
     * @enum Baud rate.
     */
    enum Baud {
        Baud460800 = 0x02,
        Baud230400 = 0x03,
        Baud115200 = 0x04,
        Baud57600  = 0x05,
        Baud28800  = 0x06,
        Baud14400  = 0x07  // Default. 
    };

    /**
     *  @enum Reset type.
     */
    enum ResetType {
        Nomal = 0x00,
        High  = 0xff
    };

    /** Constructor.
     *
     * @param tx A pin for transmit.
     * @param rx A pin for receive.
     * @param baud Baud rate. (Default is 14400.)
     */
    CameraC1098(PinName tx, PinName rx, int baud = 14400);

    /** Destructor.
     * 
     */
    ~CameraC1098();

    /** sync 
     * Make a sync. for baud rate.
     */
    ErrorNumber sync();

    /** Initialize.
     * 
     *
     * @param baud Camera Interface Speed.
     * @param jr JPEG resolution.
     */
    ErrorNumber init(Baud baud, JpegResolution jr);

    /** getJpegSnapshotPicture
     * Get JPEG snapshot picture.
     *
     * @param func A pointer to a callback function.
     *             You can block this function until saving the image datas.
     * @return Status of the error.
     */
    ErrorNumber getJpegSnapshotPicture();

    /** setmbedBaud
     *  mbed Interface Speed.
     *
     *  @param baud mbed Interface Speed.
     */
    void setmbedBaud(Baud baud);
    void Sync(void);
    void jpeg_snapshot(int CAPTURE_FRAMES);
    ErrorNumber sendReset(ResetType specialReset);  
    
private:
    SerialBuffered serial;
    static const int COMMAND_LENGTH = 6;
    static const int SYNCMAX = 60;
    static const int packageSize = 256;
    ErrorNumber sendInitial(Baud band, JpegResolution jr);
    ErrorNumber sendGetPicture(void);
    ErrorNumber sendSnapshot(void);    
    ErrorNumber sendSetPackageSize(uint16_t packageSize);
    //ErrorNumber sendReset(ResetType specialReset);  
    ErrorNumber recvData(uint32_t *length);
    ErrorNumber sendSync();
    ErrorNumber recvSync();
    ErrorNumber sendAck(uint8_t commandId, uint16_t packageId);
    ErrorNumber recvAckOrNck();

    bool sendBytes(char *buf, size_t len, int timeout_us = 20000);
    bool recvBytes(char *buf, size_t len, int timeout_us = 20000);
    bool waitRecv();
    bool waitIdle();
};

#endif
