#ifndef DTH22_h
#define DTH22_h

/* DTH22 sensor class for use with Mbed LPC1768 and other platforms
*  Copyright (c) 2013 Philip King Smith
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*
* A DTH22/RHT03 sensor data sheet is at: http://dlnmh9ip6v2uc.cloudfront.net/datasheets/Sensors/Weather/RHT03.pdf
* I made this software because the classes that others have made seemed to suffer from many errors.
* I have used my rasbpery Pi code writen in Forth as the basis for this code.
* Like the other code i have found my first attempt in Forth to get this too work suffered from the same errors!
* I have since worked out the following code that works with the idea of a run lenght timing method thus removing
*    the time sensitive error issues that seem to plauge this low cost sensor.
*/

#include "mbed.h"

#define MAXRAWDATA 1300         // the total times the DTH22 io pin is read via the mbed pin
#define DATABITS 40             // DTH22 sensor only produces 40 data bits to be read 
#define BITREADTIME 3           // the wait time on lpc1768 mbed device in us via wait_us() function
#define MAXBITCOUNT 30          // the total read cycles that when read as 1 indicate the DTH22 sensor is not sending anymore
#define STARTTRANSBITSIZE 14    // the median cycle counts of the start transmit bit from the DTH22 sensor note this is based on the BITREADTIME value
#define DATANOISE 2000          // Error indicating the DTH22 io line must be noisy and data is not valid
#define STIMINGFAIL 2001        // Error indicating the start bit timing was exceeded on some bit so data could not be properly read
#define DATANOISE2 2002         // Error indicating the DTH22 io line must be noisy and data is not valid because there was too much data from device that passed earlier error checks
#define CHECKSUMFAIL 2003       // Error indicating the DTH22 data did not pass the checksum test 

/** DTH22 Digital Temperature Humidity class
 *
 * Example:
 * @code
 * // show how the DTH22_RHT03 class works
 * #include "mbed.h"
 * #include "DTH22.h"
 * int main()
 * {
 *  int temp ;
 *  int humidity;
 *  int error ;
 *  DTH22 myhumtemp(p24);
 *  while(1) {
 *      error = myhumtemp.getTH(&temp,&humidity);
 *      printf("Temp is %ld\r\n",temp);
 *      printf("Humidity is %ld\r\n",humidity);
 *      printf("Error is %d\r\n\r\n",error);
 *      wait(2);
 *      }
 * }
 * @endcode
 */

class DTH22
{
public:
    /** Create object connected to DTH22 pin ( remember io pin needs pull up resister)
        *
        * Ensure the pull up resistors are used on this pin.
        * Any pin can be used but the pin can only be used for one DTH22 sensor and nothing else.
        *
        * @param PinName This is the pin that is connected to the IO pin of the DTH22 sensor (pin 2 of DTH22 device called Data-signal)
        */
    DTH22(PinName DATAsignal );    // Constructor

    ~DTH22();                      // Destructor

    /** Read Temperature and Humidity of DTH22 or RHT03 sensors
    *
    * This function will return when it has readings.  Note the DTH22 documents say that the device should not be read more then once every 2 seconds.
    *   This code should work with RHT03 sensors also.  The RHT03 sensor is the same as the DTH22 i believe.
    *
    * @param temp the temperature returned in this variable. Degrees celsius with one decimal so 253 is 25.3 C. (-40C to +80C)
    * @param humidity the humidity is returned in this variable. Humidity is in % rH with one decimal so 288 is 28.8 % rH. (0% to 100% rH)
    * @param returns 0 for no errors.  Any other number is some type of communication error and data might not be valid!
    */
    int getTH(int *temp,int *humidity); // Will read both temperature and humidity with an error indicator.  Note only do this read no more then once every 2 seconds as per DTH22 documents.
                                        // This will only read the DTH22 or RHT03 sensor.  To read the DTH11 sensor use the getDTH11TH member function.
                                        
    /** Read Temperature and Humidity of DTH11 sensor
    *
    * This function will return when it has readings.  Note the DTH11 documents say that the device should not be read more then once every 2 seconds.
    *   DTH11 sensor is a less percise version of the DTH22.
    *
    * @param temp the temperature returned in this variable. Degrees celsius with one decimal so 250 is 25.0 C. ( 0C to +50C)
    * @param humidity the humidity is returned in this variable. Humidity is in % rH with one decimal so 280 is 28.0 % rH. (20% to 90% rH)
    * @param returns 0 for no errors.  Any other number is some type of communication error and data might not be valid!
    */    
    int getDTH11TH(int *temp,int *humidity);    // This gets the temperature and humidity with an error indicator.  Note only do this reading no more then once every 2 seconds as per the DTH11 documents.
                                                // This will only read the DTH11 sensor.  To read the DTH22 or the RHT03 sensor use getTH member function.     

    /** Read sensor and give the raw data out for testing 
    *
    * This function will return when it has data from sensor.  Note the sensors should not be read more then once every 2 seconds.
    *
    * @param bits is a bool array that should contain 40 bits of the data from the sensor just read
    * @param data is an array of 5 bytes that the above bits get compiled into.  These bytes are simply from the above bits read from the device.
    * @param temp the temperature returned in this variable. Degrees celsius with one decimal so 253 is 25.3 C. (-40C to +80C) ( This is the DTH22 sensor value only)
    * @param humidity the humidity is returned in this variable. Humidity is in % rH with one decimal so 288 is 28.8 % rH. (0% to 100% rH) ( This is the DTH22 sensor value only)
    * @param returns 0 for no errors.  Any other number is some type of communication error and data might not be valid!
    */    
    int testing(bool *bits,signed short int *data,int *temp,int *humidity); // This will get the raw data from sensor and will produce the results as the data is processed from bits to byts to readings.
                                                                            // Note the temperature and humidity values that are returned are for the DTH22 or RHT03 sensor only but the raw data is from
                                                                            // the sensor used.  This means you can see the raw data from either the DTH22 or DTH11 sensor!

protected:
    DigitalInOut DTH22pin;
    bool rawdata[MAXRAWDATA];
    unsigned short int timingData[100];
    bool rawdatabits[DATABITS];

    void getraw();
    void getrawbits();
    int transistionCount(int index);
    int transmissionErrors();
};

#endif