/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#ifndef MBED_PWM_ALL_API_H
#define MBED_PWM_ALL_API_H

#include "device.h"



#ifdef __cplusplus
extern "C" {
#endif

/* 
SCT OUT setting
PIN     | SCT
--------------------
P0_0    | SCT0_OUT3   
P0_1    | SCT0_OUT4
P0_2    | SCT1_OUT3
P0_3    | SCT1_OUT3
P0_4    | None (SPI_0)
P0_5    | SCT0_OUT0
P0_6    | SCT2_OUT3
P0_7    | SCT0_OUT1
P0_8    | SCT0_OUT2
P0_9    | SCT1_OUT0
P0_10   | SCT1_OUT1
P0_11   | SCT1_OUT2
P0_12   | SCT2_OUT1
P0_13   | SCT2_OUT2
P0_14   | SCT1_OUT5
P0_15   | SCT3_OUT3
P0_16   | None (SPI_1)
P0_17   | SCT3_OUT1
P0_18   | SCT0_OUT5
P0_19   | SCT3_OUT2
P0_20   | SCT1_OUT6
P0_21   | None (RESET)
P0_22   | None (open drain i2c pin)
P0_23   | None (open drain i2c pin)
P0_24   | SCT0_OUT6
P0_25   | SCT2_OUT0
*/

struct pwmout_sct {
     LPC_SCT0_Type* pwm;
     uint32_t pwm_ch;
        uint32_t out_ch;
        uint32_t pin;
        uint32_t match;
};

/** Pwmout hal structure. pwmout_s is declared in the target's hal
 */
typedef struct pwmout_sct pwmout_all_t;

/**
 * \defgroup hal_pwmout Pwmout hal functions
 * @{
 */

/** Initialize the pwm out peripheral and configure the pin
 *
 * @param obj The pwmout object to initialize
 * @param pin The pwmout pin to initialize
 */
void pwmout_all_init(pwmout_all_t *obj, PinName pin);

/** Deinitialize the pwmout object
 *
 * @param obj The pwmout object
 */
void pwmout_all_free(pwmout_all_t *obj);

/** Set the output duty-cycle in range <0.0f, 1.0f>
 *
 * Value 0.0f represents 0 percentage, 1.0f represents 100 percent.
 * @param obj     The pwmout object
 * @param percent The floating-point percentage number
 */
void pwmout_all_write(pwmout_all_t *obj, float percent);

/** Read the current float-point output duty-cycle
 *
 * @param obj The pwmout object
 * @return A floating-point output duty-cycle
 */
float pwmout_all_read(pwmout_all_t *obj);

/** Set the PWM period specified in seconds, keeping the duty cycle the same
 *
 * Periods smaller than microseconds (the lowest resolution) are set to zero.
 * @param obj     The pwmout object
 * @param seconds The floating-point seconds period
 */
void pwmout_all_period(pwmout_all_t *obj, float seconds);

/** Set the PWM period specified in miliseconds, keeping the duty cycle the same
 *
 * @param obj The pwmout object
 * @param ms  The milisecond period
 */
void pwmout_all_period_ms(pwmout_all_t *obj, int ms);

/** Set the PWM period specified in microseconds, keeping the duty cycle the same
 *
 * @param obj The pwmout object
 * @param us  The microsecond period
 */
void pwmout_all_period_us(pwmout_all_t *obj, int us);

/** Set the PWM pulsewidth specified in seconds, keeping the period the same.
 *
 * @param obj     The pwmout object
 * @param seconds The floating-point pulsewidth in seconds
 */
void pwmout_all_pulsewidth(pwmout_all_t *obj, float seconds);

/** Set the PWM pulsewidth specified in miliseconds, keeping the period the same.
 *
 * @param obj The pwmout object
 * @param ms  The floating-point pulsewidth in miliseconds
 */
void pwmout_all_pulsewidth_ms(pwmout_all_t *obj, int ms);

/** Set the PWM pulsewidth specified in microseconds, keeping the period the same.
 *
 * @param obj The pwmout object
 * @param us  The floating-point pulsewidth in microseconds
 */
void pwmout_all_pulsewidth_us(pwmout_all_t *obj, int us);

/**@}*/

#ifdef __cplusplus
}
#endif


#endif
