/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "mbed_assert.h"
#include "pwm_all_api.h"
#include "cmsis.h"
#include "pinmap.h"
#include "mbed_error.h"

static LPC_SCT0_Type *SCTs[4] = {
    (LPC_SCT0_Type*)LPC_SCT0,
    (LPC_SCT0_Type*)LPC_SCT1,
    (LPC_SCT0_Type*)LPC_SCT2,
    (LPC_SCT0_Type*)LPC_SCT3,
};

// bit flags for used SCTs
static unsigned char sct_used = 0;
static int get_available_sct(void) {
    int i;
    for (i=0; i<24; i++) {
        if ((sct_used & (1 << i)) == 0)
            return i;
    }
    return -1;
}

static LPC_SCT0_Type *check_SCT_module(PinName pin){
    switch((int)pin){
        case P0_0:
        case P0_1:
        case P0_5:
        case P0_7:
        case P0_8:
        case P0_18:
            return  (LPC_SCT0_Type*)LPC_SCT0;
        
        case P0_2:
        case P0_3:
        case P0_9:
        case P0_10:
        case P0_11:
        case P0_14:
        case P0_20:
            return  (LPC_SCT0_Type*)LPC_SCT1;
        
        case P0_6:
        case P0_12:
        case P0_13:
        case P0_25:
            return  (LPC_SCT0_Type*)LPC_SCT2;
        
        case P0_15:
        case P0_17:
        case P0_19:
            return  (LPC_SCT0_Type*)LPC_SCT3;
            
    }
}

uint32_t pwm_ch_map(PinName pin){
    switch((int)pin){
        case P0_0:
        case P0_1:
        case P0_5:
        case P0_7:
        case P0_8:
        case P0_18:
            return 0;
        
        case P0_2:
        case P0_3:
        case P0_9:
        case P0_10:
        case P0_11:
        case P0_14:
        case P0_20:
            return 1;
        
        case P0_6:
        case P0_12:
        case P0_13:
        case P0_25:
            return 2;
        
        case P0_15:
        case P0_17:
        case P0_19:
            return 3;
            
    }
}

uint32_t pwm_out_map[] = {3, 4, 3, 4, 7, 0, 3, 1, 2, 0, 1, 2, 1, 2, 5, 0, 7, 1, 5, 2, 6, 7, 7, 7, 6, 0};
    
void pwmout_all_init(pwmout_all_t* obj, PinName pin) {
    MBED_ASSERT(pin != (uint32_t)NC);
    int sct_n = get_available_sct();
    
    if (sct_n == -1) {
        error("No available SCT");
    }
        
    sct_used |= (1 << sct_n);
    //obj->pwm =  SCTs[sct_n];
        obj->pwm = check_SCT_module(pin);
    
        //obj->pwm_ch = sct_n;
        obj->pwm_ch = pwm_ch_map(pin);
        
        obj->out_ch = pwm_out_map[pin];
        obj->pin = (int)pin;
        
    LPC_SCT0_Type* pwm = obj->pwm;

    // Enable the SCT clock
    LPC_SYSCON->SYSAHBCLKCTRL1 |= (1 << (obj->pwm_ch + 2));

    // Clear peripheral reset the SCT:
    //LPC_SYSCON->PRESETCTRL1 |=  (1 << (obj->pwm_ch + 2));
    //LPC_SYSCON->PRESETCTRL1 &= ~(1 << (obj->pwm_ch + 2));
        
    switch(obj->pwm_ch) {
        case 0:
            // SCT0_OUT0
            //LPC_SWM->PINASSIGN[7] &= ~0x0000FF00;
            //LPC_SWM->PINASSIGN[7] |= (pin << 8);
                
                        switch(obj->out_ch){
                            case 0:
                                LPC_SWM->PINASSIGN[7] &= ~0x0000FF00;
                                LPC_SWM->PINASSIGN[7] |= (pin << 8);
                                break;
                            
                            case 1:
                                LPC_SWM->PINASSIGN[7] &= ~0x00FF0000;
                                LPC_SWM->PINASSIGN[7] |= (pin << 16);
                                break;
                            
                            case 2:
                                LPC_SWM->PINASSIGN[7] &= ~0xFF000000;
                                LPC_SWM->PINASSIGN[7] |= (pin << 24);
                                break;
                            
                            case 3:
                                LPC_SWM->PINENABLE1 &= ~(1 << 5);
                                break;
                            
                            case 4:
                                LPC_SWM->PINENABLE1 &= ~(1 << 6);
                                break;
                            
                            case 5:
                                LPC_SWM->PINENABLE1 &= ~(1 << 7);
                                break;
                            
                            case 6:
                                LPC_SWM->PINENABLE1 &= ~(1 << 8);
                                break;
                        }
            break;
                        
        case 1:
            // SCT1_OUT0
            //LPC_SWM->PINASSIGN[8] &= ~0x000000FF;
            //LPC_SWM->PINASSIGN[8] |= (pin);
                
                        switch(obj->out_ch){
                            case 0:
                                LPC_SWM->PINASSIGN[8] &= ~0x000000FF;
                                LPC_SWM->PINASSIGN[8] |= (pin);
                                break;
                            
                            case 1:
                                LPC_SWM->PINASSIGN[8] &= ~0x0000FF00;
                                LPC_SWM->PINASSIGN[8] |= (pin << 8);
                                break;
                            
                            case 2:
                                LPC_SWM->PINASSIGN[8] &= ~0x00FF0000;
                                LPC_SWM->PINASSIGN[8] |= (pin << 16);
                                break;
                            
                            case 3:
                                LPC_SWM->PINENABLE1 &= ~(1 << 10);
                                break;
                            
                            case 4:
                                LPC_SWM->PINENABLE1 &= ~(1 << 11);
                                break;
                            
                            case 5:
                                LPC_SWM->PINENABLE1 &= ~(1 << 12);
                                break;
                            
                            case 6:
                                LPC_SWM->PINENABLE1 &= ~(1 << 13);
                                LPC_SWM->PINENABLE1 |= (1 << 22);
                                LPC_SWM->PINENABLE1 |= (1 << 23);
                                break;
                        }
            break;
                        
        case 2:
            // SCT2_OUT0
                        // LPC_SWM->PINASSIGN[8] &= ~0xFF000000;
            //LPC_SWM->PINASSIGN[8] |= (pin << 24);
            
                        switch(obj->out_ch){
                            case 0:
                                LPC_SWM->PINASSIGN[8] &= ~0xFF000000;
                                LPC_SWM->PINASSIGN[8] |= (pin << 24);
                                break;
                            
                            case 1:
                                LPC_SWM->PINASSIGN[9] &= ~0x000000FF;
                                LPC_SWM->PINASSIGN[9] |= (pin );
                                break;
                            
                            case 2:
                                LPC_SWM->PINASSIGN[9] &= ~0x0000FF00;
                                LPC_SWM->PINASSIGN[9] |= (pin << 8);
                                break;
                            
                            case 3:
                                LPC_SWM->PINENABLE1 &= ~(1 << 15);
                                break;
                            
                            case 4:
                                LPC_SWM->PINENABLE1 &= ~(1 << 16);
                                break;
                            
                        }
                        break;
        case 3:
            // SCT3_OUT0
            //LPC_SWM->PINASSIGN[9] &= ~0x00FF0000;
            //LPC_SWM->PINASSIGN[9] |= (pin << 16);
                
                    switch(obj->out_ch){
                            case 0:
                                LPC_SWM->PINASSIGN[9] &= ~0x00FF0000;
                                LPC_SWM->PINASSIGN[9] |= (pin << 16);
                                break;
                            
                            case 1:
                                LPC_SWM->PINASSIGN[9] &= ~0xFF000000;
                                LPC_SWM->PINASSIGN[9] |= (pin )<<24;
                                break;
                            
                            case 2:
                                LPC_SWM->PINASSIGN[10] &= ~0x000000FF;
                                LPC_SWM->PINASSIGN[10] |= (pin );
                                break;
                            
                            case 3:
                                LPC_SWM->PINENABLE1 &= ~(1 << 18);
                                break;
                            
                        }
            break;
        default:
            break;
    }
        
    // Unified 32-bit counter, autolimit
    pwm->CONFIG |= ((0x3 << 17) | 0x01);
    
    // halt and clear the counter
    pwm->CTRL |= (1 << 2) | (1 << 3);
    
    // System Clock -> us_ticker (1)MHz
    pwm->CTRL &= ~(0x7F << 5);
    pwm->CTRL |= (((SystemCoreClock/1000000 - 1) & 0x7F) << 5);
    
    // Match reload register
    pwm->MATCHREL0 = 20000; // 20ms
        //pwm->MATCHREL1 = (pwm->MATCHREL0 / 4); // 50% duty
        switch(obj->out_ch){
            case 0:
                pwm->MATCHREL1 = (pwm->MATCHREL0 / 4);
                break;
            
            case 1:
                pwm->MATCHREL2 = (pwm->MATCHREL0 / 4);
                break;
                        
            case 2:
                pwm->MATCHREL3 = (pwm->MATCHREL0 / 4);
                break;
                    
            case 3:
                pwm->MATCHREL4 = (pwm->MATCHREL0 / 4);
                break;
            
            case 4:
                pwm->MATCHREL5 = (pwm->MATCHREL0 / 4);
                break;
            
            case 5:
                pwm->MATCHREL6 = (pwm->MATCHREL0 / 4);
                break;
            
            case 6:
                pwm->MATCHREL7 = (pwm->MATCHREL0 / 4);
                break;
        }
    
    
        //pwm->OUT0_SET = (1 << 0); // event 0
    //pwm->OUT0_CLR = (1 << 1); // event 1
        switch(obj->out_ch){
            case 0:
                pwm->OUT0_SET = (1 << 0); // event 0
                break;
            
            case 1:
                pwm->OUT1_SET = (1 << 0); // event 0
                break;
            
            case 2:
                pwm->OUT2_SET = (1 << 0); // event 0
                break;
            
            case 3:
                pwm->OUT3_SET = (1 << 0); // event 0
                break;
            
            case 4:
                pwm->OUT4_SET = (1 << 0); // event 0
                break;
            
            case 5:
                pwm->OUT5_SET = (1 << 0); // event 0
                break;
            
            case 6:
                pwm->OUT6_SET = (1 << 0); // event 0
                break;
        }
        
        switch(obj->out_ch){
            case 0:
                pwm->OUT0_CLR = (1 << 1); // event 1
                break;
            
            case 1:
                pwm->OUT1_CLR = (1 << 2); // event 2
                break;
            
            case 2:
                pwm->OUT2_CLR = (1 << 3); // event 3
                break;
            
            case 3:
                pwm->OUT3_CLR = (1 << 4); // event 4
                break;
            
            case 4:
                pwm->OUT4_CLR = (1 << 5); // event 5
                break;
            
            case 5:
                pwm->OUT5_CLR = (1 << 6); // event 6
                break;
            
            case 6:
                pwm->OUT6_CLR = (1 << 7); // event 7
                break;
        }
        
        
    pwm->EV0_CTRL  = (1 << 12);
    pwm->EV0_STATE = 0xFFFFFFFF;
        
    //pwm->EV1_CTRL  = (1 << 12) | (1 << 0);
    //pwm->EV1_STATE = 0xFFFFFFFF;
        switch(obj->out_ch){
            case 0:
                pwm->EV1_CTRL  = (1 << 12) | (1);
                pwm->EV1_STATE = 0xFFFFFFFF;                        // event 1
                break;
            
            case 1:
                pwm->EV2_CTRL  = (1 << 12) | (2);
                pwm->EV2_STATE = 0xFFFFFFFF;                        // event 2
                break;
            
            case 2:
                pwm->EV3_CTRL  = (1 << 12) | (3);
                pwm->EV3_STATE = 0xFFFFFFFF;                        // event 3
                break;
            
            case 3:
                pwm->EV4_CTRL  = (1 << 12) | (4);
                pwm->EV4_STATE = 0xFFFFFFFF;                        // event 4
                break;
            
            case 4:
                pwm->EV5_CTRL  = (1 << 12) | (5);
                pwm->EV5_STATE = 0xFFFFFFFF;                        // event 5
                break;
            
            case 5:
                pwm->EV6_CTRL  = (1 << 12) | (6);
                pwm->EV6_STATE = 0xFFFFFFFF;                        // event 6
                break;
            
            case 6:
                pwm->EV7_CTRL  = (1 << 12) | (7);
                pwm->EV7_STATE = 0xFFFFFFFF;                        // event 6
                break;
            
        }

    // unhalt the counter:
    //    - clearing bit 2 of the CTRL register
    pwm->CTRL &= ~(1 << 2);

    // default to 20ms: standard for servos, and fine for e.g. brightness control
    pwmout_all_period_ms(obj, 20);
    pwmout_all_write    (obj, 0.5);
}


void pwmout_all_free(pwmout_all_t* obj) {
    // Disable the SCT clock
    LPC_SYSCON->SYSAHBCLKCTRL1 &= ~(1 << (obj->pwm_ch + 2));
    sct_used &= ~(1 << obj->pwm_ch);
}

void pwmout_all_write(pwmout_all_t* obj, float value) {
    LPC_SCT0_Type* pwm = obj->pwm;
    if (value < 0.0f) {
        value = 0.0;
    } else if (value > 1.0f) {
        value = 1.0;
    }
    uint32_t t_on = (uint32_t)((float)(pwm->MATCHREL0) * value);
    //pwm->MATCHREL1 = t_on;
        switch(obj->out_ch){
            case 0:
                pwm->MATCHREL1 = t_on;
                break;
            
            case 1:
                pwm->MATCHREL2 = t_on;
                break;
                        
            case 2:
                pwm->MATCHREL3 = t_on;
                break;
                    
            case 3:
                pwm->MATCHREL4 = t_on;
                break;
            
            case 4:
                pwm->MATCHREL5 = t_on;
                break;
            
            case 5:
                pwm->MATCHREL6 = t_on;
                break;
            
            case 6:
                pwm->MATCHREL7 = t_on;
                break;
        }
}

float pwmout_all_read(pwmout_all_t* obj) {
    uint32_t t_off = obj->pwm->MATCHREL0;
    uint32_t t_on  = obj->pwm->MATCHREL1;
    float v = (float)t_on/(float)t_off;
    return (v > 1.0f) ? (1.0f) : (v);
}

void pwmout_all_period(pwmout_all_t* obj, float seconds) {
    pwmout_all_period_us(obj, seconds * 1000000.0f);
}

void pwmout_all_period_ms(pwmout_all_t* obj, int ms) {
    pwmout_all_period_us(obj, ms * 1000);
}

// Set the PWM period, keeping the duty cycle the same.
void pwmout_all_period_us(pwmout_all_t* obj, int us) {
    LPC_SCT0_Type* pwm = obj->pwm;
    uint32_t t_off = pwm->MATCHREL0;
    uint32_t t_on  = pwm->MATCHREL1;
    float v = (float)t_on/(float)t_off;
    pwm->MATCHREL0 = (uint32_t)us;
    //pwm->MATCHREL1 = (uint32_t)((float)us * (float)v);
        switch(obj->out_ch){
            case 0:
                pwm->MATCHREL1 =  (uint32_t)((float)us * (float)v);
                break;
            
            case 1:
                pwm->MATCHREL2 =  (uint32_t)((float)us * (float)v);
                break;
                        
            case 2:
                pwm->MATCHREL3 =  (uint32_t)((float)us * (float)v);
                break;
                    
            case 3:
                pwm->MATCHREL4 =  (uint32_t)((float)us * (float)v);
                break;
            
            case 4:
                pwm->MATCHREL5 =  (uint32_t)((float)us * (float)v);
                break;
            
            case 5:
                pwm->MATCHREL6 = (uint32_t)((float)us * (float)v);
                break;
            
            case 6:
                pwm->MATCHREL7 = (uint32_t)((float)us * (float)v);
                break;
        }
}

void pwmout_all_pulsewidth(pwmout_all_t* obj, float seconds) {
    pwmout_all_pulsewidth_us(obj, seconds * 1000000.0f);
}

void pwmout_all_pulsewidth_ms(pwmout_all_t* obj, int ms) {
    pwmout_all_pulsewidth_us(obj, ms * 1000);
}

void pwmout_all_pulsewidth_us(pwmout_all_t* obj, int us) {
    //obj->pwm->MATCHREL1 = (uint32_t)us;
    switch(obj->out_ch){
            case 0:
                obj->pwm->MATCHREL1 = (uint32_t)us;
                break;
            
            case 1:
                obj->pwm->MATCHREL2 = (uint32_t)us;
                break;
                        
            case 2:
                obj->pwm->MATCHREL3 = (uint32_t)us;
                break;
                    
            case 3:
                obj->pwm->MATCHREL4 = (uint32_t)us;
                break;
            
            case 4:
                obj->pwm->MATCHREL5 = (uint32_t)us;
                break;
            
            case 5:
                obj->pwm->MATCHREL6 = (uint32_t)us;
                break;
            
            case 6:
                obj->pwm->MATCHREL7 = (uint32_t)us;
                break;
        }
}
