/******************************************************************************
 * @file:	main.c
 * @date:	5. jul. 2014
 * @author:	Matsuzawa Takaaki
 *----------------------------------------------------------------------------
 *
 * mbedとTOCOS-Lite Dipモジュールを使った通信のサンプルプログラム。6つのアナログ入力（加速度センサ）と
 * 4つのデジタル入力（ステータス情報）を無線で10msごと（100Hz）にTOOCOStickに送信する。
 *
 ******************************************************************************/
#include "mbed.h"

Ticker WirelessEngine;	//タイマー割り込みの宣言

/***加速度センサをアナログ入力に接続***/
/*加速度センサA*/
AnalogIn Accel_Ax(p15);	//加速度センサAのX軸
AnalogIn Accel_Ay(p16);	//加速度センサAのY軸
AnalogIn Accel_Az(p17);	//加速度センサAのZ軸
/*加速度センサB*/
AnalogIn Accel_Bx(p18);	//加速度センサBのX軸
AnalogIn Accel_By(p19);	//加速度センサBのY軸
AnalogIn Accel_Bz(p20);	//加速度センサBのZ軸
/******************************/
/***ステータスピンの設定（初期時は全てLOW（0）とする）***/
DigitalIn FlightPin(p9);	//フライトピン
DigitalIn MoveServo(p10);	//開放用サーボ
DigitalIn GachaSAT1(p11);	//ガチャサット1放出
DigitalIn GachaSAT2(p12);	//ガチャサット2放出
/******************************/
DigitalOut myled(LED1); //LED設定

Serial TOCOS(p13,p14);		//TOCOSに繋がっているシリアルピンを設定
/*
 * TOCOS ========= mbed
 * RX(p3)  <===> TX(p13)
 * TX(p10) <===> TX(p14)
 */

int i=0;

int ReadStatus(){
	/*
	 * 4本のステータスピンの情報を統合して1つの数値にして返す（ビットフラグで2進数表現）
	 * 本来、「フライトピンが抜けていないのににサーボが動作する」などの状態はありえないが、万が一の場合に備え、
	 * 全ての状態を表せるようにこのような表現方法をとっている。
	 *
	 * EX.
	 * 	＜＜状態＞＞					＜＜戻り値＞＞
	 * 	初期値(全てLOW)　				=>　	0
	 * 	フライトピン				信号HIGH	=>　	1
	 * 	フライトピン & サーボ動作	信号HIGH	=> 	3
	 * 	...
	 * 	最終状態（全てHIGH）				=> 16
	 *
	 * 			ガチャサット1		|	0	1	1	0
	 * 			ガチャサット2		|	0	0	1	1
	 * 	------------------------+-----------------
	 * 	サーボ動作ピン	|	フライトピン	|
	 * 	------------+-----------+-----------------
	 * 		0		|		0	|	0	4	12	8
	 * 		0		|		1	|	1	5	13	9
	 * 		1		|		1	|	3	7	16	11
	 * 		1		|		0	|	2	6	14	10
	 *
	 */
	return (GachaSAT2<<3|GachaSAT1<<2|MoveServo<<1|FlightPin);
}

void sendVal(){
	/*
	 * アナログピンの値をp15~p20の順にカンマ区切りで、最後にステータス情報と改行を送信
	 */
	int status = ReadStatus();	//ステータスピンの情報を読み取る

	TOCOS.printf("%d,%d,%d,%d,%d,%d,%d\n"
		,(Accel_Ax.read_u16()>>4),(Accel_Ay.read_u16()>>4),(Accel_Az.read_u16()>>4)//加速度A(X,Y,Z)
		,(Accel_Bx.read_u16()>>4),(Accel_By.read_u16()>>4),(Accel_Bz.read_u16()>>4)//加速度B(X,Y,Z)
		,status);	//ステータス 1~16の数値
}

int main() {
	TOCOS.baud(115200);	//TOCOSに繋がっているシリアルのボーレートを115200bpsに設定

	/*
	 * 0.01秒(10ms)ごとにセンサ値を取得し、TOCOSで送信する用割り込みを設定
	 * この割り込みが通信関係の処理を一定時間ごとに行ってくれるので、メインループ
	 * (while内)では別の処理を行なうことができる。
	 */
	WirelessEngine.attach(&sendVal, 0.01);

    while(1) {
    	/*
    	 * メインループ
    	 * 無線通信は割り込みで一定時間ごとに実行されるので、時間の正確さを必要としない
    	 * 処理はこのループ内に記述する。サンプルでは動作確認のため約0.2秒ごとにLED1を
    	 * 点滅させている。
    	 */

        myled = 1;
        wait(0.2);
        myled = 0;
        wait(0.2);
    }
}
