/* #############################################################################
**    Archivo        : main.c
**    Proyecto       : STM32F103C8_Barrido_Displays
**    Procesador     : STM32F103C8
**    Plataforma     : Blue Pill
**    Herramienta    : Mbed
**    Compilador     : Mbed Online C Compiler
**    Version        : Mbed-OS 5.15.0
**    Fecha/Hora     : 18-12-2019, 11:48, # CodeGen: 0
**    Descripción    :
**         Este proyecto provee lo necesario para el despliegue de números y 
**         caracteres en un display de 7 segmentos.
**         Contiene un Generador de Caracteres y las rutinas de Barrido para un
**         Display de 7 segmentos de Ánodo Común de 4 dígitos.
**         This module contains user's application code.
**   Componentes     : GPIO, Ticker, Terminal Serial.
**   Configuraciones : 
**         Modo 0.- Despliega números de hasta cuatro digitos
**         Modo 1.- Realiza el Parpadeo (Blinking) de los Puntos Decimales.
**         Modo 2.- Realiza el Parpadeo (Blinking) del valor que se esta
**                  deplegando.
**         Modo 3.- Despliega una cadena de caracteres con desplazamiento a la
**                  Izquierda.
**         Modo 3.- Despliega una cadena de caracteres con desplazamiento a la
**                  Derecha (por implementar).
**
**   Autores         :
**         ATEAM Development Group:
**          - Antulio Morgado Valle
**
**   Versión        : Beta
**   Revisión       : A
**   Release        : 0
**   Bugs & Fixes   :
**   Date           : 20/10/2019
**                    Added support for Shift Messages
**
** ###########################################################################*/
/*
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:                       I N C L U D E S
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
*/
#include "stm32f103c8t6.h"
#include "mbed.h"
//#include "AnalogIn.h"
//#include "ADC.h"
//#include "Led_RGB.h"
#include "Serial.h"
#include "TextLCD.h"
#include "I2C.h"
#include "SPI.h"
#include "stdio.h"
/*
:...............................................................................
:                       D E F I N I C I O N E S
:...............................................................................
*/
#define Led_on          0           // Estado para el Led Encendido
#define Led_off         1           // Estado para el Led Apagado
#define Buzz_On         1           // Estado para Buzzer Encendido
#define Buzz_Off        0           // Estado para Buzzer Apagado
#define ON              0           // Estado para boton presionado
#define OFF             1           // Estado para boton sin presionar
#define HOLD            2           // Estado para boton mantenido
#define RELEASE         3           // Estado para boton liberado
#define FALSE           0           // Estado FALSO
#define TRUE            1           // Estado VERDADERO
#define Ticker_Rate     1000        // Periodo de interrupción Ticker (us)
#define Scan_Time       2           // Tiempo de Barrido del Display  (ms)
#define Blink_Time      500         // Tiempo de Parpadeo del Display (ms)
#define Shift_Time      750         // Tiempo de corrimiento          (ms)
#define Event_Time      15000       // Tiempo de los eventos          (ms)
#define String_Size     20          // Tamaño de la Cadena de Caracteres
#define Digit_Null      0xFF        // Valor para apagar los Segmentos 
#define Degree_Symbol   0x0F        // Valor para el Símbolo de Grados
#define Display_Type    1           // Tipo de Display: 0 = Cátodo Común 1= Ánodo Común
#define LCD_Type        0           // Tipo Interfaz del LCD: 0=Normal, 1=I2C, 2=SPI
#define Baud_Rate       115200      // Velocidad de Transmisión (Bauds)
                                    // Velocidades Permitidas:
                                    // 300, 600, 1200, 2400, 4800, 9600,
                                    // 14400, 19600, 28800, 38400, 57600
                                    // 115200, 230400
/*
*===============================================================================
*
*           C O N F I G U R A C I O N    D E    P U E R T O S
*
*===============================================================================
*/
BusIn       Teclado (PB_6, PB_7, PB_8, PB_9);       // Definición para los Renglones: REN1,REN2,REN3,REN4
BusOut      Anodos  (PB_12, PB_13, PB_14, PB_15);   // Anodos: (Uni Dec Cent Mill)  A1 A2 A3 A4 (lsb -> msb)
BusOut      Catodos (PA_8,PA_9,PA_10,PA_11,PA_12,PA_15,PB_3,PB_4); // Cátodos: p e d c b a f g  (lsb -> msb)) 
DigitalOut  led_monitor(LED1);              // Inicializa el Led de Monitor
PwmOut      buzzer  (PB_1);                 // PWM para el Buzzer
#if LCD_Type == 0                    // Definición para pines del LCD con interfaz HD44780
DigitalOut  DB4            (PB_4);   
DigitalOut  DB5            (PB_3); 
DigitalOut  DB6            (PA_15);
DigitalOut  DB7            (PA_12);  
DigitalOut  e              (PA_11); 
DigitalOut  rs             (PA_10);
#endif
// *****    Host PC Terminal Communication channels
Serial      Terminal (PA_2, PA_3);          // Inicializa la comunicación serial a la PC (Tx,Rx)
//Serial      Terminal(USBTX, USBRX,115200);// Tx, Rx  Baud Rate - Terminal Serial por USB @ 115,200 Bauds

Ticker      Barrido;                        // Interrupción por Timer
/*
** -------------------------------------------------------------------
**    Configuración del LCD
** -------------------------------------------------------------------
*/
// El LCD puede trabajar con 3 configuraciones de la Interfaz:
// Tipo Interfaz del LCD: 
// 0=Normal, 
// 1=I2C, 
// 2=SPI
/*******************************************************************************
* I2C Communication
********************************************************************************
* Initialize the library with the numbers of the interface pins
* Board Blue Pill STM32F103C8T6
* IDE  Mbed On Line Compiler
*           I2C         PCF8574A
* Pin       Board       LCD
* (PB_10)   I2C2_SCL    SCL (white) + resistance pull-up 4.7k
* (PB_11)   I2C2_SDA    SDA (blue)  + resistance pull-up 4.7k
*  3.3V     3.3V        3.3V  (red)
*  GND      GND         GND (black)
*******************************************************************************/
#if LCD_Type == 1
I2C      i2c_lcd(PB_11,PB10);        // SDA, SCL  (I2C2)
#endif
/*******************************************************************************
* SPI Communication
********************************************************************************
* Initialize the library with the numbers of the interface pins
* Board Blue Pill STM32F103C8T6
* IDE  Mbed On Line Compiler
*           SPI             SN74HC595
* Pin       Board           LCD
*(PA_7)     SPI1_MOSI       SER   (white) 
*(PA_6)     SPI1_MISO       none  (blue)  
*(PA_5)     SPI1_CLK        SRCLK
*(PA_4)     SPI1_NSS1       RCLK
* 3.3V      3.3V            3.3V  (red)
* GND       GND             GND   (black)
*******************************************************************************/
#if LCD_Type == 2
SPI         spi_lcd(PA_7,PA_6,PA_5,PA_4);   // MOSI, MISO, SCK, CS   (SPI1)
#endif

// LCD Instantiation 
/*******************************************************************************
* Instantiation for creating a TextLCD interface 
* for using control and functions over selected port 
********************************************************************************
*/
#if LCD_Type == 0
TextLCD     lcd(PA_10,PA_11,PA_12,PA_15,PB_3,PB_4, TextLCD::LCD20x4);   // 4bit bus: rs, e, d4-d7  (lsb -> msb)
//TextLCD lcd(PB_11,PB_10,PA_7,PA_6,PA_5,PA_4, TextLCD::LCD20x2);     // 4bit bus: rs, e, d4-d7  (lsb -> msb)
#endif

#if LCD_Type == 1
TextLCD_I2C lcd(&i2c_lcd, 0x4E, TextLCD::LCD20x4);      // Address 0x4E I2C bus, PCF8574A Arduino Shield, LCD Type
//TextLCD_I2C lcd(&i2c_lcd, 0x7E, TextLCD::LCD20x4);      // Address 0x7E I2C bus, PCF8574A Arduino Shield, LCD Type
//TextLCD_I2C lcd(&i2c_lcd, 0x27, TextLCD::LCD20x4);      // Address 0x2E I2C bus, PCF8574A Arduino Shield, LCD Type
//TextLCD_I2C lcd(&i2c_lcd, 0x3F, TextLCD::LCD20x4);      // Address 0x3F I2C bus, PCF8574A Arduino Shield, LCD Type
//TextLCD_I2C lcd(&i2c_lcd, 0x7F, TextLCD::LCD20x4);      // Address 0x7F I2C bus, PCF8574A Arduino Shield, LCD Type
//TextLCD_I2C lcd(&i2c_lcd, 0x7E, TextLCD::LCD16x4);      // Address 0x3E I2C bus, PCF8574A Arduino Shield, LCD Type
//TextLCD_I2C lcd(&i2c_lcd, 0x40, TextLCD::LCD20x4);      // Address 0x40 I2C bus, PCF8574 Slaveaddress, LCD Type
//TextLCD_I2C lcd(&i2c_lcd, 0x42, TextLCD::LCD20x4);      // Address 0x42 I2C bus, PCF8574 Slaveaddress, LCD Type
//TextLCD_I2C lcd(&i2c_lcd, 0x77, TextLCD::LCD16x2);      // Address 0x77 I2C exp: I2C bus, PCF8574 Slaveaddress, LCD Type
//TextLCD_I2C lcd(&i2c_lcd, 0x40, TextLCD::LCD16x4);      // Address 0x40 I2C exp: I2C bus, PCF8574 Slaveaddress, LCD Type
//TextLCD_I2C lcd(&i2c_lcd, 0x40, TextLCD::LCD20x4);      // Address 0x42 I2C exp: I2C bus, PCF8574 Slaveaddress, LCD Type
//TextLCD_I2C lcd(&i2c_lcd, 0x42, TextLCD::LCD16x2, TextLCD::WS0010);  // I2C exp: I2C bus, PCF8574 Slaveaddress, LCD Type, Ctrl Type
#endif

#if LCD_Type == 2
TextLCD_SPI lcd(&spi_lcd, PA_4, TextLCD::LCD16x4, TextLCD::HD44780);// SPI bus, SN74595 expander, CS pin, LCD Type 
//TextLCD_SPI lcd(&spi_lcd, PA_4, TextLCD::LCD20x4, TextLCD::HD44780);// SPI bus, CD4094 expander, CS pin, LCD Type 
//TextLCD_SPI_N lcd(&spi_lcd, PB_12, TextLCD::LCD16x2, NC, TextLCD::ST7032_3V3); // SPI native: SPI bus, CS pin, RS pin, LCDType=LCD16x2, BL=NC, LCDTCtrl=ST7032
//TextLCD_SPI lcd(&spi_lcd, PA_4, TextLCD::LCD24x4D);      // SPI bus, CD4094 expander, CS pin, LCD Type
#endif

/* END Port Declarations */
/*
#===============================================================================
|
|                           A R E A   D E   D A T O S
|
#===============================================================================
*/
/*
+-------------------------------------------------------------------------------
|  Variables Globales de Usuario 
+-------------------------------------------------------------------------------
*/
uint16_t Rate=Ticker_Rate/2;    // Velocidad de barrido (500us = 0.5ms)
// ++++++++++++++++++++ Generador de Caracteres ++++++++++++++++++++++++++++++++
//             Offset:  0    1    2    3    4    5    6    7    8    9   10    11  12   13   14   15   16   17   18   19   20   21  22    23   24   25   26   27   28   29   30   31   32   33   34   35   36   37   38   39   40   41   42   43   44   45   46   47  48   49   50   51   52   53   54   55   56   57   58   59   60   61   62   63   64   65   66   67   68   69   70   71   72   73   74   75   76   77   78   79   80   81   82   83   84   85   86   87   88   89   90   91  92
//Caracter a Desplegar: 0    1    2    3    4    5    6    7    8    9    A     b   C    d    E    f    G    h    I    j    K    L   M     n    o    P    q    r    S    t    u    V    W    X    y    Z  space a    e    /e   @    ñ    [    ]    U    /U   /_   -    _   _/_  /=   =   ==    °   +/c   +n   +/c  +u   o    c    n    /c   u    ^    ¬   + >   ^   / v   <    v    >   i    +i   /i   +/i  ii   +ii  #   /#    J    /J   -J   -/J  |    ||     |  +    /+   F    +F   /F   +/F  null         !  "   #   $  %   &   '   (    )   *   +   -   .   /   :   ;   <   >   ?   @
uint8_t  Segmentos[]={0x81,0xE7,0x49,0x43,0x27,0x13,0x11,0xC7,0x01,0x03,0x05,0x31,0x99,0x61,0x19,0x1D,0x91,0x35,0xBD,0xE3,0x2D,0xB9,0x85,0x75,0x71,0x0D,0x07,0x7D,0x13,0x39,0xF1,0xB1,0xA1,0x25,0x23,0x49,0xFF,0x41,0x09,0x03,0x11,0x55,0x99,0xC3,0xA1,0x85,0xDF,0x7F,0xFB,0xDB,0x7B,0x5F,0x5B,0x0F,0x1F,0x8F,0x4F,0x2F,0x71,0x79,0x75,0x73,0xF1,0x3F,0x9F,0xCF,0x6F,0x7D,0xF9,0xF3,0x77,0xFD,0xBF,0xF7,0xEF,0xF5,0xAF,0x6D,0x37,0xE3,0xB9,0x9D,0xC7,0xBD,0xA5,0xE7,0x3D,0x67,0x1D,0x47,0x63,0x39,0x00};
// ++++++++++++++++++++++ Selector de Digitos ++++++++++++++++++++++++++++++++++
uint8_t  Digitos[]={0x08,0x04,0x02,0x01,0x00,0x0F};     // Millares, Centenas, Decenas, Unidades, Apagados, Encendidos
uint8_t  Numeros[]={1,2,3,4};   // Valor a Desplegar
uint8_t  Cadena[]={33,0,21,10,36,14,28,18,22,14,36,35,10,12,10,29,14,23,12,24,36,36,36}; // Mensaje a Desplegar
uint8_t  punto_decimal=0;       // Posición del punto decimal: 0=sin punto, 1=unidades, 2=decenas, 3=centemas 4=millares,  
uint8_t  cont_1ms;              // Contador de milésimas de Segundo
uint8_t  cont_10ms;             // Contador de centésimas de Segundo
uint8_t  cont_100ms;            // Contador de décimas de Segundo
uint8_t  cont_1s;               // Contador de Segundos
uint16_t scan_rate =Scan_Time ; // Tiempo de barrido para el Display (ms)
uint16_t shift_rate=Shift_Time; // Tiempo del Corrimiento del Display (ms)
uint16_t blink_rate=Blink_Time; // Tiempo de Parpadeo (ms)
uint16_t event_rate=Event_Time; // Tiempo entre Eventos (ms)
uint8_t  Secuencia=0;           // Secuencia a ejecutar
uint8_t  i=0;                   // Incice de Barrido
uint8_t  corrimiento=0;         // Contador de Corrimientos
bool     mode=0;                // Modo de despliegue: 0=Normal, 1=Corrimiento
bool     flag1=FALSE;           // Bandera para Barrido Display
bool     flag2=FALSE;           // Bandera para Corrimientos
bool     flag3=FALSE;           // Bandera para Parpadeo
bool     flag4=FALSE;           // Bandera para Secuenciador
bool     Blinking=FALSE;        // 0=Sin Parapadeo, 1=Parpadeo
/*
++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
|  Definición de Funciones Prototipo y Rutinas de los Vectores de Interrupción
++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*/
void Barrido_OnInterrupt(void); 
void Clear_Screen(void); 
void Buzzer_Beep(void);

/* END definitions */ 

/*
#===============================================================================
|
|               P R O G R A M A     P R I N C I P A L
|
#=============================================================================== 
*/
int main()          
{                           // Inicialización de variables, puertos e interrupciones
confSysClock();             // Configure system clock (72MHz HSE clock, 48MHz USB clock)
Barrido.attach_us(&Barrido_OnInterrupt, Ticker_Rate);   // Le asigna el periodo de barrido de 1ms (Rate=1000)
Terminal.baud(Baud_Rate);   // Se configura la velocodad de transmisión   
Terminal.printf("\e[2J");   // Clear Screen
Terminal.printf(" System is Wake Up!.\n\r");    //e inicia la comunicación a la Terminal  
Buzzer_Beep();              // Prueba el Buzzer
Clear_Screen();             // Limpa la Pantalla de la Terminal Serial  
Terminal.printf(" System is Ready to Work...\n\r");    // e inicia la  comunicación serial
Terminal.printf ("\t\e[1;37;41mInstituto Politecnico Nacional\e[0m\n\r");
Terminal.printf ("\e[1;37;42mESCUELA SUPERIOR DE INGENIERIA MECANICA Y ELECTRICA\e[0m\n\r\v");
Terminal.printf ("\t\e[1;34;40mBarrido \e[1;37;40mdel \e[1;32;40mDisplay \e[1;33;40mde \e[1;31;40m7 Segmentos\e[0m\n\r\v\v\e[0m"); 
Terminal.printf ("\nSeleccion del Modo de Operacion:\n\r");
  while (TRUE)              // El Lazo del Programa principal está aquí !!!
  {  
    if (Secuencia==0)
    {
      Terminal.printf ("Modo 0 = Normal (4 Digitos)    \r");            
    }
    if (Secuencia==1)
    {
      Terminal.printf ("Modo 1 = Parpadeo Puntos       \r");        
    }   
    if (Secuencia==2)
    {
      Terminal.printf ("Modo 2 = Parpadeo Digitos      \r");
    }
    if (Secuencia==3)
    {
      Terminal.printf ("Modo 3 = Corrimiento Izquierda \r"); 
    }
    if (Secuencia==4)
    {
      Terminal.printf ("Modo 4 = Corrimiento Derecha   \r"); 
    }           
    if (flag1==TRUE)        // Está puesta la Bandera 1 ?
    {                         
  //----------------------------------------------------------------------------
  //----------------------- BARRIDO DE LOS DÍGITOS -----------------------------
  //----------------------------------------------------------------------------
  //............................................................................ 
  // ...................... Display de Ánodo Común .............................
  //............................................................................ 
  #if Display_Type == 1
      Anodos= Digitos[4];           // Apaga todos los Digitos (Evita fantasmas)                
      Catodos= Segmentos[Numeros[i]];  // Actualiza los segmentos
      if (i==4-punto_decimal) Catodos=(Segmentos[Numeros[i]]& 0xFE); // Enciende el punto decimal
      Anodos= Digitos[i];           // Activa el Digito en cuestión
  //----------------------- Parpadeo de los Puntos -----------------------------
      if (Secuencia==1)             // Si estamos en modo Normal:
      {
        if ( Blinking) punto_decimal=3;  // Activa los puntos del Reloj
//        if ( Blinking && i==2) punto_decimal=2;  // Activa los puntos del Reloj (para LTC-5650) 
        if (!Blinking) punto_decimal=0;  // Desactiva los puntos del Reloj
        if (i==4-punto_decimal) Catodos=(Segmentos[Numeros[i]]& 0xFE); // Enciende el punto decimal
        Anodos= Digitos[i];                 // Activa el Digito en cuestión
      }  
  //----------------------- Parpadeo de los Digitos ----------------------------
      if (Secuencia==2 && Blinking == 1) Anodos= Digitos[4]; // Apaga todos los Digitos
  //----------------------- Corrimiento de los Digitos -------------------------
      if (Secuencia==3)             // Está en modo 3? (corrimiento de mensaje)
      {
        Numeros[i]=Cadena[i+corrimiento]; // Copia 4 caracteres al buffer de despliegue
        if (corrimiento >= String_Size) corrimiento=0;   // Restablece el contador de caractres
        if (i==4)                   // Se terminó de barrer los digitos? 
        {
          i=0;                      // Restablece el índice de barrido
//          mode=0;                 // Restablece el Modo Normal 
        }                         
      }           
  #endif
  //............................................................................          
  // ...................... Display de Cátodo Común ............................
  //............................................................................ 
  #if Display_Type == 0
      Anodos= ~(Digitos[4]);          // Apaga todos los Digitos (Evita fantasmas)                
      Catodos= ~(Segmentos[Numeros[i]]);    // Actualiza los segmentos
      if (i==4-punto_decimal) Catodos=~(Segmentos[Numeros[i]]& 0xFE); // Enciende el punto decimal
      Anodos= ~(Digitos[i]);                // Activa el Digito en cuestión
      if (mode==2) Anodos= ~(Digitos[4]);   // Apaga todos los Digitos
      if (mode==1)                          // Si estamos en modo Normal
      {
        if ( Blinking) punto_decimal=3;  // Activa los puntos del Reloj
//        if ( Blinking && i==2) punto_decimal=2;  // Activa los puntos del Reloj (para LTC-5650)         
        if (!Blinking) punto_decimal=0;  // Desactiva los puntos del Reloj
        if (i==4-punto_decimal) Catodos=~(Segmentos[Numeros[i]]& 0xFE); // Enciende el punto decimal
        Anodos= ~(Digitos[i]);              // Correspondiente al Digito en cuestión
      }             
  #endif
  //............................................................................

  //----------------------- Actualiza Indices ----------------------------------
      i++;                      // Apunta al siguiente dígito
      if (i==4)                 // Es el últino dígito?
      {
        i=0;                    // Reinicia el Barriodo
      }      
      flag1=FALSE;              // Apaga la Bandera 1
    }
    //--------------------- Corrimiento de los Digitos -------------------------
    if (flag2==TRUE)            // Está puesta la Bandera 2?
    {
      if (Secuencia==3)
      {
        mode=3;                 // Tiempo entre desplazamientos del mensaje
        corrimiento++;          // corrimiento al siguiente conjunto de de caracteres
      }   
      shift_rate = Shift_Time;  // Restablece el tiempo de los corrimientos
      flag2=FALSE;              // Limpia la Bandera 2
    }

    //--------------------- Blinking o Parpadeo --------------------------------   
    if (flag3==TRUE)            // Está puesta la Bandera 3?
    {
      led_monitor=!led_monitor; // parapadeo del led_monitor
      flag3=FALSE;              // Limpia la Bandera 3
    } 
    //--------------------- Secuenciación de Eventos ---------------------------
    if (flag4==TRUE)            // Está puesta la Bandera 4?
    {
       if (Secuencia==0)        // Reinica las Secuencias
       {
         mode=0;                // Nos vamos a modo Normal
       }        
       if (Secuencia==1)        // Si la secuencoa es Puntos Parapadeantes
       {
         mode=1;                // Nos vamos a modo Parpadeo de Puntos
       }
       if (Secuencia==2)        // Si la secuencoa so Digitos Parapadeantes
       {
         mode=2;                // Nos vamos a modo Parpadeo de los Dígitos
         i=0;                   // Reinicia el indice de los dígitos
       }  
       if (Secuencia==3)        // Si la secuencoa es Corrimiento a la Izquierda
       {
         mode=3;                // Nos vamos a modo de Corrimiento a la Izquierda
       } 
       Secuencia++;
       if (Secuencia >= 4)      // Si se terminaron las secuencias
       {             
         corrimiento=0;         // Restablece el buffer de despliege
         i=0;                   // Reinicia el indice de los dígitos
           Secuencia = 0;       // Reinicia las secuencias
       }
       flag4=FALSE;             // Limpia la Bandera 4
    }
  }
}
/* END main */

/*
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::              V E C T O R E S    D E    I N T E R R U P C I Ó N
::                      (Rutinas de Atención)
::
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
*/
/* #############################################################################
**    Evento       : Barrido_OnInterrupt()
**    Proyecto     : STM32F103C8_Barrido_Display
**    Procesador   : STM32F103C8T6
**    Herramienta  : Mbed
**    Compilador   : Mbed Online C Compiler
**    Version      : Mbed-OS 5.15.0
**    Compiler     : GNU C Compiler
**    Date/Time    : 18-12-2019, 11:48, # CodeGen: 0
**    Abstract     :
/*!
**     @brief
**         Called after Ticker is done, [Interrupt service/event]
**         and it is enabled: 
**         TimerInt.attach_us(&TimerInt_OnInterrupt, Ticker_Rate);
**         the event is enabled and will be executed every  [Ticker_Rate]
**         This is user's event module.
**         Put your event handler code here.
**     @param
**         UserDataPtr     - Pointer to the user or
**                           RTOS specific data. The pointer is passed
**                           as the parameter of Init method.
**    Settings    :
**    Contents    :
**         Ticker - &Barrido_OnInterrupt;
**
** ###########################################################################*/
void Barrido_OnInterrupt()          // Rutina de Atención al Ticker
{
    cont_1ms++;                     // Incrmenta el contador de milisegundos
    scan_rate--;                    // Decrementa el tiempo de  despliege
    shift_rate--;                   // Decrementa el tiempo de desplazamiento del display
    event_rate--;                   // Decrementa el contador de eventos
    blink_rate--;                   // Decrementa el tiempo de parpadeo
    if(!scan_rate)                  // Se acabo el tiempo de desplige?
    {
        scan_rate=Scan_Time;        // Se restable el tiempo de despliegue     
        flag1=TRUE;                 // Enciende la bandera para las rutinas de despliege
    }        
  //-------------------- Corrimiento de la Pantalla ----------------------------

    if (!shift_rate)
    {
        shift_rate=Shift_Time;      // Restablece el contador de corrimientos
        flag2=TRUE;            
    }
  //---------------------- Parpadeo del Display --------------------------------

    if (!blink_rate)
    {
        Blinking= !Blinking;        // Parpadeo
        blink_rate=Blink_Time;      // Restablece el contador de parpadeo
        flag3=TRUE;            
    } 
  //--------------------- Secuenciador de Eventos ------------------------------

    if (!event_rate)
    {   
        event_rate=Event_Time;      // Restablece el contador de eventos
        flag4=TRUE;            
    }        
  //--------------------------- Reloj ------------------------------------------
    
    if (cont_1ms>=10)               // Son 10ms?
    {
        cont_10ms++;                // Incrementa el contador de decimas de ms
        cont_1ms=0;                 // Restablece el contador de ms
    }      
    if (cont_10ms>=10)              // Son 100ms?
    {
        cont_100ms++;               // Incrementa el contador de centecimas de ms
        cont_10ms=0;                // Restablece el contador de decimas de seg
    }   
    if (cont_100ms>=10)             // Son 1000ms o sea 1 seg?
    {
        cont_1s++;                  // Incrementa el contador de segundos
        cont_100ms=0;               // Restablece el contador de centesimas de seg
    }    
    if (cont_1s>=10)                // Son 10seg?
    {
        cont_1s=0;                  // Restablece el contador de segundos
    }
}  
/* END Events */ 
/*
 ______________________________________________________________________________
/______________________________________________________________________________\
|                                                                              |
|                   F U N C I O N E S     P R O T O T I P O                    |
|______________________________________________________________________________|
 \_____________________________________________________________________________/
*/
                // Las Funciones Prototipo van aquí !

/*
|-------------------------------------------------------------------------------
|     Función     :     Clear_Screen()
|      
|     Descripción :     Limpia la Pantalla de la Terminal
|
|     Parámetros  :     Ninguno
|          
|     Retorna     :     Nada
|-------------------------------------------------------------------------------
*/
void Clear_Screen() 
{
    // ANSI Terminal Commands
    Terminal.printf("\x1B[2J");     // secuencia de Escape  \e1B[2J
    Terminal.printf("\x1B[H");      // secuencia de Escape  \e1B[H
}

/*
|-------------------------------------------------------------------------------
|     Función     :     Buzzer_Beep()
|      
|     Descripción :     Rutina para prueba del Buzzer
|
|     Parámetros  :     Ninguno
|          
|     Retorna     :     Nada
|-------------------------------------------------------------------------------
*/
void Buzzer_Beep()                      // Software routine for Buzzer Beep
{ 
  Terminal.printf("Testing Buzzer... \n\r");
  buzzer = Buzz_On;
  wait_ms(750);
  buzzer = Buzz_Off;
  wait_ms(125);
  buzzer = Buzz_Off;   
  wait_ms(125);
  buzzer = Buzz_On;
  wait_ms(500);
  buzzer = Buzz_Off;
  Terminal.printf("Buzzer OK ! \n\r");
}
/* END Functions */                 
/* END Routines */                   
/* END program */

/*
*+******************************************************************************      
*+              La documentación va aquí!:
*+******************************************************************************      
*+
*+------------------------------------------------------------------------------
*+ Descripción del Display de 7 Segmentos de Anodos Comunes en modo multiplexado
*+------------------------------------------------------------------------------
*+                             14 13 12 11 10 9  8
*+                             |  |  |  |  |  |                         
*+                             |  |  |  |  |  |  
*+               --------------------------------------------------
*+              |               A  a  f  A  A  b                   |
*+              |               4        3  2                      |
*+              |  ______________________________________________  |             
*+              | |       ____      ____      _____     _____    | |
*+              | |     /     /   /     /    /     /   /     /   | |
*+              | |    /____ /   /____ / .  /____ /   /____ /    | |
*+              | |   /     /   /     / .  /     /   /     /     | |
*+              | |  /____ /.  /_____/ .  /____ /.  /____ /.     | |
*+              | |                                              | |
*+              | |______________________________________________| |
*+              |                                                  |
*+              |                              A                   |
*+              |               e  d  p  c  g  1                   |
*+               --------------------------------------------------
*+                              |  |  |  |  |  |  |                       
*+                              |  |  |  |  |  |  |
*+                              1  2  3  4  5  6  7                                  
*+
*+                              Display: HS410561K-D30
*+
*+
*+                             14 13 12 11 10 9  8
*+                             |  |  |  |  |  |                         
*+                             |  |  |  |  |  |  
*+               --------------------------------------------------
*+              |               A  g  f  p  a  b                   |
*+              |               4                                  |
*+              |  ______________________________________________  |             
*+              | |       ____      ____      _____     _____    | |
*+              | |     /     /   /     /    /     /   /     /   | |
*+              | |    /____ /   /____ / .  /____ /   /____ /    | |
*+              | |   /     /   /     / .  /     /   /     /     | |
*+              | |  /____ /.  /_____/    /____ /.  /____ /.     | |
*+              | |                                              | |
*+              | |______________________________________________| |
*+              |                                                  |
*+              |                  A  A        A                   |
*+              |               e  3  2  c  d  1                   |
*+               --------------------------------------------------
*+                              |  |  |  |  |  |  |                       
*+                              |  |  |  |  |  |  |
*+                              1  2  3  4  5  6  7                                  
*+
*+                               Display: BQ-M544RE
*+
*+
*+                            16 15 14 13    11 
*+                             |  |  |  |    |                           
*+                             |  |  |  |    |    
*+               --------------------------------------------------
*+              |              b  g  a  c    f                     |
*+              |                                                  |
*+              |  ______________________________________________  |             
*+              | |       ____      ____      _____  .  _____    | |
*+              | |     /     /   /     /    /     /   /     /   | |
*+              | |    /____ /   /____ / .  /____ /   /____ /    | |
*+              | |   /     /   /     / .  /     /   /     /     | |
*+              | |  /____ /.  /_____/ .  /____ /.  /____ /.     | |
*+              | |                                              | |
*+              | |______________________________________________| |
*+              |                                                  |
*+              |               A  A     A     A     A             |
*+              |               4  3  d  p  e  2  p  1             |
*+               --------------------------------------------------
*+                              |  |  |  |  |  |  |  |                      
*+                              |  |  |  |  |  |  |  |
*+                              1  2  3  4  5  6  7  8                                  
*+
*+                              Display: KEM-3942BSR
*+
*+
*+                            12 11  10 9  8  7
*+                             |  |  |  |  |  |                           
*+                             |  |  |  |  |  |    
*+               --------------------------------------------------
*+              |              A  a  f  A  A  b                    |
*+              |              4        3  2                       |
*+              |  ______________________________________________  |             
*+              | |       ____      ____      _____  .  _____    | |
*+              | |     /     /   /     /    /     /   /     /   | |
*+              | |    /____ /   /____ / .  /____ /   /____ /    | |
*+              | |   /     /   /     / .  /     /   /     /     | |
*+              | |  /____ /.  /_____/ .  /____ /.  /____ /.     | |
*+              | |                                              | |
*+              | |______________________________________________| |
*+              |                                                  |
*+              |                              A                   |
*+              |               e  d  p  c  g  1                   |
*+               --------------------------------------------------
*+                              |  |  |  |  |  |  
*+                              |  |  |  |  |  |             p1 = g
*+                              1  2  3  4  5  6             p2 = c             
*+                                                           p3 = d
*+                              Display: HS-5463BS           p4 = e
*+                              Cátodo Común                 pu = b
*+                                                           pd = f
*+
*+
*+                              14 13 12 11 10 8 9
*+                                 |  |  |  |  |                           
*+                                 |  |  |  |  |    
*+               --------------------------------------------------
*+              |                  A  a  f  A  A                   |
*+              |                  4        3  2                   |
*+              |  ______________________________________________  |             
*+              | |       ____      ____    . _____  .  _____    | |
*+              | |     /     /   /     /    /     /   /     /   | |
*+              | |    /____ /   /____ /    /____ /   /____ /    | |
*+              | |   /     /   /     /    /     /   /     /     | |
*+              | |  /____ /.  /_____/ .  /____ /   /____ /      | |
*+              | |                                              | |
*+              | |______________________________________________| |
*+              |                                                  |
*+              |                              A                   |
*+              |               e  d  p  c  g  1  b                |
*+               --------------------------------------------------
*+                              |  |  |  |  |  |  |                        
*+                              |  |  |  |  |  |  |  
*+                              1  2  3  4  5  6  7                                    
*+
*+                              Display: LTC-5650G
*+
+*
*+                            12 11 10  9  8  7
*+                             |  |  |  |  |  |                         
*+                             |  |  |  |  |  |  
*+               --------------------------------------------------
*+              |              A  a  f  A  A  b                    |
*+              |              4        3  2                       |
*+              |  ______________________________________________  |             
*+              | |       ____      ____      _____     _____    | |
*+              | |     /     /   /     /    /     /   /     /   | |
*+              | |    /____ /   /____ / .  /____ /   /____ /    | |
*+              | |   /     /   /     / .  /     /   /     /     | |
*+              | |  /____ /.  /_____/ .  /____ /.  /____ /.     | |
*+              | |                                              | |
*+              | |______________________________________________| |
*+              |                                                  |
*+              |                              A                   |
*+              |               e  d  p  c  g  1                   |
*+               --------------------------------------------------
*+                              |  |  |  |  |  |                        
*+                              |  |  |  |  |  |  
*+                              1  2  3  4  5  6                                    
*+
*+                              Display: FYQ-2841BH
*+                              Display: FYQ-3641BS
*+                              Display: FYQ-5641BS
*+
*+
*+                      PB_15 PB_14 PB_13 PB_12 
*+                        A4    A3    A2    A1
*+                       Mill  Cent  Dec   Uni
*+ 
*+     STM32-F103C8      PB_4  PB_3 PA_15 PA_12   PA_11 PA_10  PA_9  PA_8    #    Hex     Offset
*+            a
*+         ______         g     f     a     b       c     d     e     p       
*+        /     /         1     0     0     0       0     0     0     1      0    0x81      0
*+     f /  g  / b        1     1     1     0       0     1     1     1      1    0xE7      1
*+      /_____/           0     1     0     0       1     0     0     1      2    0x49      2
*+     /     /            0     1     0     0       0     0     1     1      3    0x43      3
*+  e /     / c           0     0     1     0       0     1     1     1      4    0x27      4
*+   /_____/ .            0     0     0     1       0     0     1     1      5    0x13      5
*+      d    p            0     0     0     0       0     0     1     1      6    0x11      6
*+                        1     1     0     0       0     1     1     1      7    0xC7      7
*+                        0     0     0     0       0     0     0     1      8    0x01      8
*+                        0     0     0     0       0     0     1     1      9    0x03      9
*+                        0     0     0     0       0     1     0     1      A    0x05      10
*+                        0     0     1     1       0     0     0     1      b    0x31      11
*+                        1     0     0     1       1     0     0     1      C    0x99      12
*+                        0     1     1     0       0     0     0     1      d    0x61      13
*+                        0     0     0     1       1     0     0     1      E    0x19      14
*+                        0     0     0     1       1     1     0     1      f    0x1D      15
*+                        1     0     0     1       1     0     0     1      G    0x91      16
*+                        0     0     1     1       0     1     0     1      h    0x35      17
*+                        1     0     1     1       1     1     0     1      i    0xBD      18
*+                        1     1     1     0       0     0     1     1      J    0xE3      19
*+                        0     0     1     0       1     1     0     1      k    0x2D      20
*+                        1     0     1     1       1     0     0     1      L    0xB9      21
*+                        1     0     0     0       0     1     0     1      M    0x85      22
*+                        0     1     1     1       0     1     0     1      n    0x75      23
*+                        0     1     1     1       0     0     0     1      o    0x71      24
*+                        0     0     0     0       1     1     0     1      P    0x0D      25
*+                        0     0     0     0       0     1     1     1      q    0x07      26
*+                        0     1     1     1       1     1     0     1      r    0x7D      27
*+                        0     0     0     1       0     0     1     1      s    0x13      28
*+                        0     0     1     1       1     0     0     1      t    0x39      29
*+                        1     1     1     1       0     0     0     1      u    0xF1      30
*+                        1     0     1     1       0     0     0     1      V    0xB1      31
*+                        1     0     1     0       0     0     0     1      w    0xA1      32
*+                        0     0     1     0       0     1     0     1      X    0x25      33
*+                        0     0     1     0       0     0     1     1      y    0x23      34
*+                        0     1     0     0       1     0     0     1      Z    0x49      35
*+                        1     1     1     1       1     1     1     1   space   0xFF      36
*+                        0     1     0     0       0     0     0     1      a    0x41      37
*+                        0     0     0     0       1     0     0     1      e    0x09      38
*+                        0     0     0     0       0     0     1     1     /e    0x03      39
*+                        0     0     0     1       0     0     0     1      @    0x11      40
*+                        0     1     0     1       0     1     0     1      ñ    0x55      41
*+                        1     0     0     1       1     0     0     1      [    0x99      42
*+                        0     0     0     0       1     0     0     1      ]    0xC3      43
*+                        1     0     1     0       0     0     0     1      U    0xA1      44
*+                        1     0     0     0       0     1     0     1     /U    0x85      45
*+                        1     1     0     1       1     1     1     1     /_    0xDF      46
*+                        0     1     1     1       1     1     1     1      -    0x7F      47
*+                        1     1     1     1       1     0     1     1      _    0xFB      48
*+                        1     1     0     1       1     0     1     1    _/_    0xDB      49
*+                        0     1     1     1       1     0     1     1     /=    0x7B      50
*+                        0     1     0     1       1     1     1     1      =    0x5F      51
*+                        0     1     0     1       1     0     1     1     ==    0x5B      52
*+                        0     0     0     0       1     1     1     1      °    0x0F      53
*+                        1     1     1     1       0     0     0     1    + c    0xF1      38
*+                        1     1     1     1       0     1     0     0    +/c    0xF4      39
*+                        1     0     1     0       0     1     1     1      c    0xA7      40
*+                        1     0     0     1       0     1     1     1     /c    0x97      41
*+                        1     1     1     1       0     0     1     0    + u    0xF2      42
*+                        1     1     1     1       1     0     0     0    + n    0xF8      43
*+                        1     0     1     0       1     0     0     1      [    0xA9      44
*+                        1     0     0     1       1     1     0     0      ]    0x9C      45
*+                        1     1     1     1       0     0     0     0      °    0xF0      46
*+                        1     1     1     1       1     1     0     0      ¬    0xFC      47
*+                        1     1     1     1       0     1     1     0    + >    0xF6      48
*+                        1     1     1     0       0     1     1     1     ^     0xE7      49
*+                        1     0     1     0       1     1     1     1    / <    0xAF      50
*+                        1     0     0     1       1     1     1     1    / v    0x9F      51
*+                        1     1     0     1       0     1     1     1      >    0xD7      52
*+                        1     1     1     1       0     0     1     1      v    0xF3      53
*+                        1     1     1     1       1     0     0     1      <    0xF9      54
*+                        1     1     1     1       1     1     0     1    / _    0xBF      55
*+                        1     1     1     1       0     1     1     1      -    0xF7      56
*+                        1     0     1     1       1     1     1     1      _    0xFD      57
*+                        1     1     1     1       0     1     0     1      =    0xF5      58
*+                        1     0     1     1       0     1     1     1    / =    0xB7      59
*+                        1     0     1     1       0     1     0     1     ==    0xB5      60
*+                        1     0     1     1       1     1     0     1     _¬    0xBD      61
*+                        1     1     0     0       1     1     1     1     ii    0xCF      62
*+                        1     1     1     1       1     0     1     0      #    0xFA      63
*+                        1     0     0     1       1     1     1     0      j    0x9E      64
*+                        1     1     0     1       1     1     0     0     /j    0xDC      65
*+                        1     1     1     0       1     0     0     1    - j    0xE9      66
*+                        1     0     1     0       1     0     1     1    -/j    0xAB      67
*+                        1     0     0     0       1     1     1     0    - J    0x8E      68
*+                        1     1     0     1       1     0     0     0    -/J    0xD8      69
*+                        1     1     1     0       1     0     1     1     |     0xEB      70
*+                        1     1     0     0       1     0     1     0     ||    0xCA      71
*+                        1     1     0     1       0     1     1     0      +    0xD6      72
*+                        1     1     1     0       0     0     1     1     /+    0xE3      73
*+                        1     0     0     1       1     0     1     1  espacio  0xFF      74
*+
*+ Nota: Cada dígito se activa al hacer la señal A correspondiente en "alto"
*+       y enviar las señales de los segmentos que se desean encender en "bajo" 
*+
**+-----------------------------------------------------------------------------
*+        CD4041   Quad True/Complement Buffer
*+------------------------------------------------------------------------------
*+
*+             |\    1
*+             | \-------  A1
*+          3  |  \
*+ PB_12  -----|   >
*+             |  /  2
*+             | /o------ REN1
*+             |/
*+
*+
*+             |\    11
*+             | \-------  A2
*+          13 |  \
*+ PB_13  -----|   >
*+             |  /  12
*+             | /o------- REN2
*+             |/
*+
*+
*+             |\    4
*+             | \-------  A3
*+          6  |  \
*+ PB_14  -----|   >
*+             |  /  5
*+             | /o------ REN3
*+             |/
*+
*+
*+
*+             |\    8
*+             | \-------  A4
*+          10 |  \
*+ PB_15  -----|   >
*+             |  /  9
*+             | /o------ REN4
*+             |/
*+
*+
*+----------------------------------------------------------------------------*/
/* END documentation */ 
/*
+-------------------------------------------------------------------------------
|                               EJERCICIO
|
| 1.-   Crear la función para convertir cualquier cantidad o valor entero a 
|       codigo BCD y representarlo en un buffer para su despliegue en un formato
|       de Millares, Centenas, Decenas y Unidades dentro de un arreglo para su 
|       despliegue en el display de 7 segmentos.
| 2.-   Crear la función para convertir cualquier valor ASCII a codigo BCD 
|       y representarlo en un buffer de un arreglo de Millares, Centenas, 
|       Decenas y unidades para su despliegue en el display de 7 segmentos.
| 3.-   Implementar la rutina para eliminación de los ceros a la izquierda 
|       (Blanking).
| 4.-   Implementar el menú para seleccionar entre Modo 0 y Modo 1 por el usuario

|
+------------------------------------------------------------------------------+
|
|                               EXAMEN
|
| 1.- Modificar este programa para: 
|       a) Leer una cadena de caracteres (Mensaje) desde la terminal serial.
|       b) Convertir todos los caracteres de la cadena a Mayúsculas (o minúsculas).
|       c) Depositarlas en un Buffer para su posterior despliegue en el Display 
| 2.- Utilizando el generador de caracteres para el display 
|       a) Traducir los caracteres de la cadena a su equivalente del generador 
|          de caracteres para ser desplegados por el display de 7 segmentos.
|       b) Crear la cadena traducida en el buffers de despliegue y adicionar al
|          final 4 espacios (para indicar el fin de la cadena) 
|       c) Delimitar el tamaño del buffer de despliegue para que se ajuste a la
|          longitud de la cadena a desplegar más 4 espacios. Este ajuste debe 
|          ser automático y ajustarse de acuerdo al tamaño del mensaje 
|          corresponedientes para depositar las nuevas cadenas de mensajes
|          generados y poderlas desplegar.  
|       d) Desplegar dicha cadena mediante corrimientos a la izquierda en el 
|          Display de 4 Dígitos.
|      (La longitud para el Buffer para despliegue no debe exceder los 80 
|      caracteres).
|       
|  Tip.- Consultar la Tabla ASCII 
|        (Para los números, restarles 0x30 y para las letras, restarles 0x37)
|        (Para la Ñ, restar 0x7C y para el espacio, sumarle 0x04)
|
| 3.- Utilizando el Convertidor Analógico/Digital.
|       a) Obtener una medición en representación flotante (0.0 a 1.0) y escalarla
|          para poder ser representada en formato BCD de 4 dígitos (0-9999).
|       b) proveer rutinas para poder escalar y hacer ajustes por medio del punto 
|          decimal.
|
+------------------------------------------------------------------------------+
*+   Powered by
*+
*+          T H E     A N T U L I U ' S   T E A M   R&D  Unltd 
*+
*+  Research, Development, Systems, Equipment, Support & Spare Parts.
*+  (C) Copyright 1989-2019, All Rights Reserved            
*+
*+  Welcome to The Beaners Proyect Community!                   
*+----------------------------------------------------------------------------*/
/* END Mbed */ 