/*-----------------------------------------------------------
 * F469_SLIDER Library v1.0
 * Copyright (c) 2018 Wynand Steenberg
 * s216875730@mandela.ac.za
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *-----------------------------------------------------------
 */

#ifndef F469_SLIDER_HPP
#define F469_SLIDER_HPP

#include "mbed.h"
#include <string>
#include "TS_DISCO_F469NI.h"
#include "LCD_DISCO_F469NI.h"
#include "F469GUI.hpp"

/** A Class library for using Sliders on the DISCO-F469NI Development board.  The class
 * uses the existing BSP class created by Team ST.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "F469_SLIDER.hpp"
 * 
 * DigitalOut led_green(LED1);
 * DigitalOut led_orange(LED2);
 * DigitalOut led_red(LED3);
 * DigitalOut led_blue(LED4);
 * 
 * TS_DISCO_F469NI ts_;
 * LCD_DISCO_F469NI lcd_;
 * 
 * 
 * char char_SliderDisplayValue[4];    // String Value to display for the Slider Control value 
 * uint16_t SliderDisplayValue_;       // Variable used to access Slider Control Value in F469SLIDER.cpp
 * 
 * int main()
 * {
 *     led_green = 1;         // Switch off all LEDs
 *     led_orange = 1;
 *     led_red = 1;
 *     led_blue = 1; 
 * 
 *     lcd_.Clear(LCD_COLOR_WHITE);                                        // Set LCD Background colour
 *    
 *     Slider sld1(lcd_, ts_, 20, 350, 220, 40, 5, 1, 20,      
 *                  LCD_COLOR_BLUE, LCD_COLOR_YELLOW, 1, "Slider1", Font20);   // Define sld1 slider            
 *     sld1.Render();                                                          // Draw sld1 Slider  
 *    
 *     Slider Slider2(lcd_, ts_, 300, 350, 300, 40, 50, 1, 100,      
 *                  LCD_COLOR_BLUE, LCD_COLOR_YELLOW, 2, "Slider2", Font20);   // Define Slider2 slider            
 *     Slider2.Render();                                                       // Draw Slider2 Slider      
 *        
 *     lcd_.SetTextColor(LCD_COLOR_BLACK);                                                         // Set Text colour to Black
 *     lcd_.SetFont(&Font20);                                                                      // Font size 20
 *     lcd_.DisplayStringAt(5, 5, (uint8_t *)"Slider driver for DISCO_F469", LEFT_MODE);  // Display main header text
 * 
 *     while (true)                                                    // Main program loop
 *     {
 *         led_green = 1;                                              // Switch off all LEDs
 *         led_orange = 1;
 *         led_red = 1;
 *         led_blue = 1; 
 *         
 *         if (sld1.Moved())                                           // Check if sld1 Slider was touched and run instructions if true
 *         {    
 *             led_blue = 0; 
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);                     
 *             lcd_.FillRect(85, 265, 80, 40);                         // Draw border to display Slider Control Value
 *             lcd_.SetTextColor(LCD_COLOR_WHITE);
 *             lcd_.FillRect(95, 270, 60, 30);                         // Draw border to display Slider Control Value                                     
 *             
 *             sprintf(char_SliderDisplayValue, "%3d", (int) SliderDisplayValue_);                     // Convert integer to text
 *             lcd_.SetFont(&Font16);                                                                  // Set Font size
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);                                                     // Set Font colour
 *             lcd_.DisplayStringAt(100, 280, (uint8_t *)char_SliderDisplayValue, LEFT_MODE);          // Write Slider Control Value to the LCD
 *             
 *             if (SliderDisplayValue_ < 10)                                                           // Slider Control Value decision
 *                 {
 *                     lcd_.DisplayStringAt(60, 330, (uint8_t *)"                     ", LEFT_MODE);   // Overwrite previous text
 *                     lcd_.DisplayStringAt(60, 330, (uint8_t *)"SLIDER1 < 50%", LEFT_MODE);           // Write text to LCD
 *                 }
 *             else if (SliderDisplayValue_ == 10)                                                     // Slider Control Value decision
 *                 {  
 *                     lcd_.DisplayStringAt(60, 330, (uint8_t *)"                     ", LEFT_MODE);   // Overwrite previous text
 *                     lcd_.DisplayStringAt(60, 330, (uint8_t *)"SLIDER1 = 50%", LEFT_MODE);           // Write text to LCD
 *                 }
 *             else                                                                                    // Slider Control Value decision
 *                 {  
 *                     lcd_.DisplayStringAt(60, 330, (uint8_t *)"                     ", LEFT_MODE);   // Overwrite previous text
 *                     lcd_.DisplayStringAt(60, 330, (uint8_t *)"SLIDER1 > 50%", LEFT_MODE);           // Write text to LCD
 *                 }          
 *         }                                                                                           // End sld1 instructions
 * 
 *         if (Slider2.Moved())                                      // Check if Slider2 Slider was touched and run instructions if true
 *         {    
 *             led_green = 0; 
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);                     
 *             lcd_.FillRect(400, 265, 80, 40);                         // Draw border to display Slider Control Value
 *             lcd_.SetTextColor(LCD_COLOR_WHITE);
 *             lcd_.FillRect(410, 270, 60, 30);                         // Draw border to display Slider Control Value                                     
 *             
 *             sprintf(char_SliderDisplayValue, "%3d", (int) SliderDisplayValue_);                     // Convert integer to text
 *             lcd_.SetFont(&Font16);                                                                  // Set Font size
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);                                                     // Set Font colour
 *             lcd_.DisplayStringAt(415, 280, (uint8_t *)char_SliderDisplayValue, LEFT_MODE);          // Write Slider Control Value to the LCD
 *         }    
 *  
 *         wait(0.02f);   
 *     }                                                               // End Main program loop
 * }                                                                   // End Main program 
 * @endcode
 */

    class Slider
    {
    public:
        //! Constructor
        Slider(LCD_DISCO_F469NI &lcd, TS_DISCO_F469NI &ts,
               uint16_t x, uint16_t y, uint16_t width, uint16_t height,
               uint16_t value_C, uint16_t min, uint16_t max,
               uint32_t sliderColour, uint32_t changeColour,
               int style, const string label = "", sFONT &fonts = Font12,
               uint32_t textColour = LCD_COLOR_WHITE)
               : lcd_(lcd), ts_(ts), SliderX_(x), SliderY_(y), SliderW_(width), SliderH_(height),
                 valueControl_(value_C), Min_ (min), Max_ (max), 
                 SLIDER_COLOUR_(sliderColour), CHANGE_COLOUR_(changeColour),
                 STYLE_(style), LABEL_(label), FONTS_(&fonts), FONT_WIDTH_(fonts.Width),
                 FONT_HEIGHT_(fonts.Height), active_(true)
                 { Render(); }

        /**
          * @brief  Draw Slider.
          *         
          */
        void Render();
        
        /**
          * @brief  Service Slider.
          *         
          */
        void Service();
               
        /**
          * @brief  Check touch detected on Slider.
          *         
          */
        bool Moved();
   
        /**
          * @brief  Check if touch is on Slider.
          *         
          */
        bool SliderBoundaryCheck();
        
        /**
          * @brief  Check previous state of Slider.
          *         
          */
        static TS_StateTypeDef GottenState()
        {   return state_; }

        /**
          * @brief  Set or reset multi-touch.
          *         
          */
        static void SetMultiTouch(bool tf) { multiTouch = tf; }

                
    private:
        LCD_DISCO_F469NI &lcd_;
        TS_DISCO_F469NI &ts_;

        const uint16_t SliderX_, SliderY_, SliderW_, SliderH_, valueControl_, Min_, Max_;
        const uint32_t SLIDER_COLOUR_;  // original color
        const uint32_t CHANGE_COLOUR_;  // colour to change to color
        const int STYLE_;               // Slider style        
        const string LABEL_;            // Slider label
        sFONT *const FONTS_;
        const uint16_t FONT_WIDTH_;
        const uint16_t FONT_HEIGHT_;
        bool active_;                   // Slider active boolean
        
        static TS_StateTypeDef state_;
        static bool multiTouch;

        /**
          * @brief  Disallow copy constructor and assignment operator.
          *         
          */
        Slider(const Slider&);
        Slider& operator=(const Slider&);
    };
#endif  // F469_SLIDER_HPP
