#ifndef TRIAL_H_
#define TRIAL_H_

#include "arraylist.h"
#include "trialtime.h"
#include "random.h"

#include "task.h"

struct TrialFlag {
    bool cued;
    bool responded;
    bool rewarded; // only used during Pair mode
    bool reset;
    
    TrialFlag(): cued(false), responded(false), rewarded(false), reset(false) {}
    
    void clear();
    void writeToSerial(const Task& task);
};

struct Trial {
    
    /**
    *   the trial count used during the Pair/Report/Associate modes
    */
    uint8_t      index;
    
    /**
    * whether the animal whisked during the cue.
    * + whether the animal waited for the cue.
    */
    TrialFlag     flag;
    
    /**
    * the timestamp when the trial started.
    */
    trialtime_t   starting;
    /**
    * the timestamp when the cue started
    */
    trialtime_t   cuestarting;
    
    /**
    * the total waiting period for the animal during this trial before the cue.
    */
    unsigned long waiting;   // used for calculation of waiting period
    
    /**
    * the duration of the delay period for this trial.
    */
    uint16_t      delay_dur_ms;
    
    // (passive) visual cue-related param in the Associate mode.
    uint64_t      vis_onset_us;
    
    Trial(): index(1) { }
    
    /**
    *   initialize the trial parameters according to the given Task parameter set.
    *   typically called from one of the automaton states.
    */
    void reset(const Task& task);
    
    /**
    *   a helper function to assign onset & duration randomly for the (passive) visual cue.
    *
    *   the `onset` will distribute exponentially from `0` to `auddur - respdur - mindur`.
    *   the `duration` will distribute uniformly from `mindur` to `phasedur - onset`.
    */
    void assignAssociative(const Task& task);
    
    void assignForFeedback(const Task& task);
    
    /**
    *   a callback mechanism for marking (timestamping) the start of the trial.
    *   called from one of the automaton states.
    */
    void markTrialStart();
    
    /**
    *   a callback mechanism for marking (timestamping) the end of the waiting period of the trial.
    *   called from one of the automaton states.
    */
    void markEndOfWait();
    
    /**
    *   a callback mechanism for marking (timestamping) the end of the trial.
    *   called from one of the automaton states.
    */
    void markTrialEnd(const Task& task);
    
    /**
    *   output the result of a trial into a serial port through the IO mechanism.
    *   typically called from one of the automaton states.
    */
    void writeToSerial(const Task& task);
};

#endif