#include "task.h"
#include "utils.h"
#include "states.h"
#include "automaton.h"

Task::Task(const Mode& mode):
    mode(mode),
    delay_min_ms        (CFG_DELAY_MIN), 
    delay_var_ms        (CFG_DELAY_VAR),
    prep_dur_ms         (CFG_PREP_DUR),
    aud_dur_ms          (CFG_AUD_DUR),
    aud_tick_hz         (CFG_AUD_FREQ),
    resp_dur_ms         (CFG_RESP_DUR),
    post_dur_ms         (CFG_POST_DUR),
    reward_dur_ms       (CFG_REWARD_DUR),
    onset_min_ms        (CFG_ONSET_MIN),
    onset_steps_n       (CFG_ONSET_STEPS),
    vis_dur_ms          (CFG_VIS_DUR),
    vis_test_every      (CFG_VIS_TEST_EVERY),
    whisk_debounce_ms   (CFG_WHISK_DEBOUNCE),
    lick_debounce_ms    (CFG_LICK_DEBOUNCE),
    test_reward         (CMD_TEST_REWARD, callback(this, &Task::testReward)      ),
    clear_index         (CMD_CLEAR_INDEX, callback(this, &Task::clearTrialIndex) ),
    run                 (CMD_EXECUTE,     callback(this, &Task::runTrial)        )
{ }

const char ModeSelection::CMD_ID_MODE        = '_';
const char ModeSelection::CMD_MODE_PAIR      = 'P';
const char ModeSelection::CMD_MODE_STAGE     = 'S';
const char ModeSelection::CMD_MODE_REPORT    = 'R';
const char ModeSelection::CMD_MODE_ASSOCIATE = 'A';
const char ModeSelection::CMD_MODE_MOTION    = 'M';
const char ModeSelection::CMD_MODE_MOTION_ALT= 'N';

ModeSelection::ModeSelection(const Mode& defaultValue):
    value(defaultValue) {
    config::addCommand(CMD_ID_MODE, this);
}

ModeSelection::~ModeSelection() { 
    config::removeCommand(CMD_ID_MODE);
}

bool ModeSelection::parse(const char& c) {
    switch(c) {
    case CMD_MODE_PAIR:         value = Pair; return true;
    case CMD_MODE_REPORT:       value = Report;    return true;
    case CMD_MODE_STAGE:        value = Stage;    return true;
    case CMD_MODE_ASSOCIATE:    value = Associate; return true;
    case CMD_MODE_MOTION:       value = Motion;    return true;
    case CMD_MODE_MOTION_ALT:   value = MotionAlt; return true;
    default:
        return false;
    }
}

bool ModeSelection::writeSettings() {
#define WRITE(CHR, VAL) if (value == (VAL)) { IO::write("[%c]",CHR); } else { IO::write("%c",CHR); }
    WRITE(CMD_MODE_PAIR,        Pair)
    WRITE(CMD_MODE_REPORT,      Report)
    WRITE(CMD_MODE_STAGE,       Stage)
    WRITE(CMD_MODE_ASSOCIATE,   Associate)
    WRITE(CMD_MODE_MOTION,      Motion)
    WRITE(CMD_MODE_MOTION_ALT,  MotionAlt)
#undef WRITE
    return true;
}

void ModeSelection::echoback() { 
    IO::write(IO::CONFIG_HEADER);
    writeSettings();
    IO::write("\r\n");
}


void Task::testReward() {
    automaton::init<TestReward>();
    automaton::run();
    IO::result("out");
}

void Task::clearTrialIndex() {
    automaton::init<ClearTrialIndex>();
    automaton::run();
    IO::result("cleared");
}
 
void Task::runTrial() {
    automaton::init<Delay>();
    automaton::run();
}

void Task::parseFromSerial() {
    config::handleSerial();
}

void Task::writeSettingsToSerial() {
    config::writeSettingsToSerial();
}
