#ifndef PULSE_H_
#define PULSE_H_

#include "mbed.h"


/**
*   implementation for turning on the port
*   for a specified duration (using a timer).
*
*   the state of a Pulse instance follows the
*   sequence below, after calling the run() method:
*
*   Rest   -(user calling: run)->   Armed   (output=L) 
*   Armed  -(after onset:  start)-> Active  (output=H)
*   Active -(after dur:    stop)->  Rest    (output=L)
*
*   Although you can generate a force-transition to the specified state
*   by calling e.g. start() or stop(), it is not a recommended usage.
*/
class Pulse
{
public:
    enum Status {
        Armed,
        Active,
        Interrupted,
        Rest
    };

    Pulse(PinName pin, const uint64_t& onset_us=0, const uint64_t& duration_us=1000, const uint64_t& blink_us=0);
    
    void setEnabled(const bool& value);
    bool isEnabled();
    void setOnset(const uint64_t& value_us);
    void setDuration(const uint64_t& value_us);
    void setBlinkDuration(const uint64_t& vaue_us);
    
    void attachTurnOnCallback(Callback<void ()> cb=0);
    void attachTurnOffCallback(Callback<void ()> cb=0);
    void detachTurnOnCallback();
    void detachTurnOffCallback();
    
    /**
    *   runs the pulse output.
    *
    *   the state of this Pulse can be retrieved via getStatus().
    */
    void run();
    
    /**
    *   force-start the output.
    */
    void start();
    
    /**
    *   aborts the output.
    */
    void stop();
    
    /**
    *   use this output as a normal DigitalOut
    */
    void direct(const bool& value);
    
    Status getStatus();
    
    /**
    *   busy-wait until its status becomes from Active to Rest.
    *   if the status is already at Rest, then it returns immediately. 
    */
    void wait();
    
    void loginfo();

private:
    void blink();

    DigitalOut  out_;
    Timeout     timer_;
    Ticker      blinker_;
    bool        enabled_;
    uint64_t    onset_;
    uint64_t    dur_;
    uint64_t    blinkdur_;
    Callback<void ()> turnon_;
    Callback<void ()> turnoff_;
    
    volatile Status      stat_;
};

#endif
