#include "pulse.h"
#include "IO.h"

Pulse::Pulse(PinName pin,
             const uint64_t& onset_us,
             const uint64_t& duration_us,
             const uint64_t& blink_us):
    out_(pin, 0),
    enabled_(true),
    stat_(Rest)
{
    setOnset(onset_us);
    setDuration(duration_us);
    setBlinkDuration(blink_us);
}

void Pulse::setEnabled(const bool& value)
{
    enabled_ = value;
}

void Pulse::setOnset(const uint64_t& value_us)
{
    onset_ = value_us;
}

bool Pulse::isEnabled()
{
    return enabled_;
}

void Pulse::setDuration(const uint64_t& value_us)
{
    dur_ = value_us;
}

void Pulse::setBlinkDuration(const uint64_t& value_us)
{
    blinkdur_ = value_us;
}

void Pulse::loginfo() {
    const uint32_t onset_ms = onset_/1000;
    const uint32_t duration_ms = dur_/1000;
    IO::debug("pulse: enabled=%d, onset=%d ms, duration=%d ms", enabled_, onset_ms, duration_ms);
}

void Pulse::attachTurnOnCallback(Callback<void ()> cb)
{
    turnon_ = cb;
}

void Pulse::attachTurnOffCallback(Callback<void ()> cb)
{
    turnoff_ = cb;
}

void Pulse::detachTurnOnCallback()
{
    turnon_ = 0;
}

void Pulse::detachTurnOffCallback()
{
    turnoff_ = 0;
}

void Pulse::run()
{
    if (!enabled_) return;
    
    if (onset_ < 500) {
        // do not wait for the onset timeout
        start();
    } else {
        stat_ = Armed;
        timer_.attach_us(callback(this, &Pulse::start), onset_);
    }
}

void Pulse::start()
{
    if (!enabled_) return;
    
    stat_ = Active;
    out_.write(1);
    timer_.attach_us(callback(this, &Pulse::stop), dur_);
    if (blinkdur_ > 0) {
        blinker_.attach_us(callback(this, &Pulse::blink), blinkdur_);
    }
    if (turnon_) {
        turnon_();
    }
}

void Pulse::stop()
{
    if (!enabled_) return;
    
    timer_.detach();
    blinker_.detach();
    
    out_.write(0);
    stat_ = Rest;
    if (turnoff_) {
        turnoff_();
    }
}

void Pulse::blink()
{
    out_ = !out_;
}

void Pulse::direct(const bool& value) {
    out_.write(value? 1:0);
}

Pulse::Status Pulse::getStatus()
{
    return stat_;
}

void Pulse::wait()
{
    while(stat_ != Rest);
}

    